#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <time.h>

#define GFP_KTHREAD

extern void *kmalloc(uint16_t size, int flags);
extern void kfree(void *mem);
extern int kmalloc_init(void);

#define DEBUG_RUN
#ifndef DEBUG_RUN
#define dbg_print(...) do{}while(0)
#else
#define dbg_print(...) fprintf(stderr, __VA_ARGS__)
#endif

#define MEM_OBJS 256
struct {
	void *mem;
	int size;
} allocated[MEM_OBJS];
int total_alloced = 0;
#define TEST_RAM_SIZE 0x900
extern uint8_t test_ram[0x900];
#ifdef DEBUG_RUN
uint8_t last_test_ram[0x900];
#endif

void dump_test_ram(unsigned char *arg) {
	int i;
	int pos;
	for (i = 0x100; i < TEST_RAM_SIZE; i++) {
		pos = i % 32;
		if (pos == 0) {
			fprintf(stderr, "%03x: ", i);
		}
		fprintf(stderr, "%02x", arg[i]);
		if (pos == 31) {
			int j;
			fprintf(stderr, " |");
			for (j = i - 31; j <= i; j++) {
				char ch = arg[j];
				if (ch == 0)
					ch = ' ';
				else if (ch <= ' ' || ch >= 0x7e)
					ch = '.';
				fprintf(stderr, "%c", ch);
				if (j % 32 == 15)
					fprintf(stderr, "|");
			}
			fprintf(stderr, "|\n");
		} else if (pos == 15)
			fprintf(stderr, "-");
		else
			fprintf(stderr, " ");
	}
	fprintf(stderr, "\n");
}

void test_chain(void) {
	unsigned short mblk;
	int flag, size, last_flag = 0x8000;
	int last_addr = 0;
	int i;
	for (i = 0x100; i < 0x900; ) {
		mblk = test_ram[i] + (test_ram[i+1] << 8);
		size = mblk & 0x7fff;
		flag = mblk & 0x8000;
// 		fprintf(stderr, "test_chain mblk@%03x: %04x\n", i, mblk);
		if (!flag && !last_flag) {
			fprintf(stderr, "ERROR: 2 adjacent free blocks! %02x %02x\n", last_addr, i);
#ifdef DEBUG_RUN
			dump_test_ram(last_test_ram);
			dump_test_ram(test_ram);
#endif
			exit(1);
		}
		last_flag = flag;
		i += size + 2;
	}
	if (i != 0x900) {
		fprintf(stderr, "ERROR: Block list does not end at 0x900! %x\n", i);
#ifdef DEBUG_RUN
		dump_test_ram(last_test_ram);
		dump_test_ram(test_ram);
#endif
		exit(1);
	}
}


void test_allocs(void) {
	int i;

	for (i = 1; i < MEM_OBJS; i++) {
		int size;
		unsigned char *ptr;
		int j;
		if (!allocated[i].mem)
			continue;
		ptr = allocated[i].mem;
		size = allocated[i].size;
		for (j = 0; j < size; j++) {
			if (ptr[j] != i) {
				fprintf(stderr, "ERROR: Object %02x is corrupted @%02x: %02x!, ramaddr=%x\n", i, j, ptr[j], (int)(ptr - test_ram));
#ifdef DEBUG_RUN
				dump_test_ram(last_test_ram);
				dump_test_ram(test_ram);
#endif
				exit(1);
			}
		}
	}
}

void run_tests(void) {
	test_chain();
	test_allocs();
}

void free_all(void) {
	int i;

	for (i = 1; i < MEM_OBJS; i++) {
		int size;
		unsigned char *ptr;
		int j;
		if (!allocated[i].mem)
			continue;
		kfree(allocated[i].mem);
		allocated[i].mem = NULL;
		total_alloced -= allocated[i].size;
	}
	run_tests();
}

int allocations = 0;

int main(int argc, char * argv[]) {
	int i;
	int init_vector;

	if (argc == 2)
		init_vector = atoi(argv[1]);
	else
		init_vector = time(NULL);

	dbg_print("init_vector=%d\n", init_vector);
	srand(init_vector);
	kmalloc_init();
	dbg_print("Init done. test_ram=%p\n", test_ram);
	run_tests();
// 	dump_test_ram();
	for (i = 0; i < MEM_OBJS * 10000; i++) {
// 		if (i % 100 == 0)
// 			free_all();
		int obj = rand() % MEM_OBJS;
		if (obj == 0)
			continue;
		if (allocated[obj].mem) {
			dbg_print("Freeing obj %02x size %3d offset %x\n", obj, allocated[obj].size, allocated[obj].mem - (void*)test_ram);
			memset(allocated[obj].mem, 0, allocated[obj].size);
			kfree(allocated[obj].mem);
			if (((unsigned char*)allocated[obj].mem)[-1] & 0x80) {
				fprintf(stderr, "ERROR: Freed block has no free marker!\n");
			}
			allocated[obj].mem = NULL;
			total_alloced -= allocated[obj].size;
			dbg_print("Freeing done.\n");
		} else {
			int size = rand() % 256;
			void *mem;
			int flags = 0;
			if (size == 0) // FIXME: Allocation of size 0 should return NULL!
				continue;
			if (size > 64)
				flags = 1;
			dbg_print("Allocating obj %02x size %3d\n", obj, size);
			mem = kmalloc(size, flags);
			if (mem) {
// 				if (flags)
// 					dbg_print("Successfully allocated kthread\n");
				dbg_print("Allocated offset %x\n", mem - (void*)test_ram);
				total_alloced += size;
				allocations++;
				allocated[obj].mem = mem;
				allocated[obj].size = size;
				memset(mem, obj, size);
			} else {
				dbg_print("Allocation failed\n");
// 				continue;
			}
		}
		run_tests();
#ifdef DEBUG_RUN
		memcpy(last_test_ram, test_ram, 0x900);
#endif
// 		dump_test_ram();
	}
	run_tests();
	fprintf(stderr, "Successful! %d allocations %d bytes occupied\n", allocations, total_alloced);
	dump_test_ram(test_ram);
	free_all();
	dump_test_ram(test_ram);
	return 0;
}

