// SPDX-License-Identifier: GPL-2.0
#include <linux/printk.h>
#include <linux/sched.h>
#include <linux/task.h>
#include <linux/init.h>
#include <linux/kernel.h>

#include <rupsched/avr232.h>

#include <asm/system.h>
#include <asm/io.h>
#include <asm/gpio.h>
#include <linux/timer.h>
#include <linux/delay.h>
#include <asm/ioregs.h>


// HRPC: High resolution performance counter
#ifndef _HRPC_CH
#define _HRPC_CH

static int init_hrpc(void) {
	if (0) { // Enable interrupt
		out(TIMSK1, TIMSK1_TOIE1);
	}
	out(TCCR1B, TCCR1B_CS1_PS_1);
	return 0;
}
early_initcall(init_hrpc);

static inline u16 read_hrpc(void) {
	return in_16(TCNT1);
}

/* Interrupt latency measurement:
 * The measured interrupt is the timer interrupt, which needs to save all registers.
 * - Blank loop (without interrupts)  16 cycles
 * - With common interrupt handling, -Os OPTIMIZE_SIZE, SMALL_SP: 172
 * - With Classic interrupts: 150
 * - With optimization for speed -O3, OPTIMIZE_SIZE=n, LARGE_SP: 127
 * - The same as above, but with -O2: 143 cycles
 * - same as above, Common interrupt handling: 175
 * - with -O3: 172
 * - back to classic interrupts: 127
 * - with LARGE_SP: 127 (the same). This means SMALL_SP has no runtime cost.
 *   Of course, we don't measure task switch performance.
 * - with -Os: 127 (no runtime cost wtf)
 */

#endif /* _HPET_CH */


struct task_list {
	struct task_struct *first;
	struct task_struct *last;
};
extern struct task_list prios[CONFIG_MAX_PRIOS];

#ifdef CONFIG_STACK_SIZE_FIXED
static struct task_struct usart_task[1];
static struct task_struct usart_task_a[1];
static struct task_struct usart_task_b[1];
#else
static struct task_struct *usart_task;
static struct task_struct *usart_task_a;
static struct task_struct *usart_task_b;
#endif

extern void dump_memory(void);

void show_task_list(void) {
	int i;
	struct task_struct *t;
	printk(KERN_NOTICE "a->next=%x b->next=%x\n", usart_task_a->next, usart_task_b->next);
	for (i = 0; i < CONFIG_MAX_PRIOS; i++) {
		printk(KERN_NOTICE "p[%d] %x %x\n", i, prios[i].first, prios[i].last);
		for (t = prios[i].first; t; t = t->next) {
			printk(KERN_NOTICE " T{%x}=", t);
			printk(KERN_NOTICE "{ %x, %x, %d, %d }\n", t->entity.prio, t->next, t->entity.bonus, t->entity.weight);
		}
	}
}

struct wake_sleep_struct {
	struct timer timer;
	struct completion completion;
};

static void wake_sleep_completion(struct timer *timer) {
	struct wake_sleep_struct *ws = (struct wake_sleep_struct*)timer;
	complete_from_irq(&ws->completion);
}
static void sleep(int seconds) {
	struct wake_sleep_struct wake_sleep;

	init_completion(&wake_sleep.completion);

	wake_sleep.timer.expires = jiffies + HZ * seconds;
	wake_sleep.timer.function = wake_sleep_completion;
	add_timer(&wake_sleep.timer);
	wait_for_completion(&wake_sleep.completion);
}

u16 init_cycles;

__kthread_func static void usart_thread(void *arg) {
	char ch;
	int ch_num;
// 	u16 now, tdiff;
// 	now = read_hrpc();
// 	now2
	

	local_irq_disable();
	printk(KERN_NOTICE "Hello World from task!\n");
// 	printk(KERN_NOTICE "arg=%x\n", arg);
// 	printk(KERN_NOTICE "sp=%x\n", in_16(SP));
	local_irq_enable();
	
	local_irq_enable();
	for (;;) {
		
		ch_num = avr232_getc();
		if (ch_num < 0) {
			continue;
		}
// 		printk(KERN_NOTICE " got %02x\n", ch_num);
		ch = ch_num;
#ifdef CONFIG_PANIC
		if (ch == 'D') {
			dump_memory();
			continue;
		}
		if (ch == 'P') {
			panic("test");
		}
#endif
		if (ch == 'T') {
			local_irq_disable();
			show_task_list();
			local_irq_enable();
		}
		if (ch == 't') {
			u24 i;
			u16 t, last_t;
			u16 max_t = 0;
			u16 tdiff;
			last_t = read_hrpc();
			for (i = 0; i < 1000000; i++) {
				t = read_hrpc();
				tdiff = t - last_t;
				if (tdiff > max_t)
					max_t = tdiff;
				last_t = t;
			}
			printk(KERN_NOTICE "max delay=%u\n", max_t);
		}
		if (ch == 'W') {
			local_irq_disable();
			mdelay(1000);
			local_irq_enable();
		}
		if (ch == 'S') {
// 			printk(KERN_NOTICE "sleep_completion.done=%d\n", sleep_completion.done);
			printk(KERN_NOTICE "Waiting for a second...\n");
			sleep(1);
			printk(KERN_NOTICE "done.\n");
		}
		avr232_putc(ch);
		if (ch == '\r') {
			ch = '\n';
			avr232_putc('\n');
#ifdef CONFIG_STACK_SIZE_REPORTING
			report_stack_usage();
#endif
		}
	}
}

// #define VERBOSE_TASKS
// #define DELAYED_TASKS

__kthread_func static void usart_a_thread(void *arg) {
	int i = 0;
	printk(KERN_NOTICE "TASK A\n");
	for (;;) {
#ifdef DELAYED_TASKS
		mdelay(500);
#endif
#ifdef VERBOSE_TASKS
		i++;
		if (i == 20) {
			local_irq_disable();
			avr232_putc('\r');
			avr232_putc('\n');
			local_irq_enable();
			i = 0;
		}
		local_irq_disable();
		avr232_putc('A');
		local_irq_enable();
#endif
		gpio_out(GPIO_D7, 0);
// 		gpio_out(GPIO_C0, 1);
	}
}

__kthread_func static void usart_b_thread(void *arg) {
	int i = 0;
	printk(KERN_NOTICE "TASK B\n");
	for (;;) {
#ifdef DELAYED_TASKS
		mdelay(500);
#endif
#ifdef VERBOSE_TASKS
		i++;
		if (i == 20) {
			local_irq_disable();
			avr232_putc('\n');
			avr232_putc('\r');
			local_irq_enable();
			i = 0;
		}
		local_irq_disable();
		avr232_putc('B');
		local_irq_enable();
#endif
		gpio_out(GPIO_D7, 1);
// 		gpio_out(GPIO_C0, 0);
	}
}

#if CONFIG_MAX_PRIOS < 3
#error Need to use prio 2
#endif
#ifndef CONFIG_SCHED_OTHER
#error Need CONFIG_SCHED_OTHER
#endif

static int init_usart_sched_timer_test(void) {
// 	out(DDRD, 7);
	gpio_set_output(GPIO_D7);

	printk(KERN_NOTICE "\n\n\nHello World from init! HZ=%d\n", HZ);
#ifdef CONFIG_STACK_SIZE_FIXED
	kernel_thread(usart_task, usart_thread, (void*)0x1234, 2);
	kernel_thread(usart_task_a, usart_a_thread, (void*)0x1234, -1);
	kernel_thread(usart_task_b, usart_b_thread, (void*)0x1234, -2);
#else
	kernel_thread(128, usart_thread, (void*)0x1234, 2);
	kernel_thread(64, usart_a_thread, (void*)0x1234, -1);
	kernel_thread(64, usart_b_thread, (void*)0x1234, -2);
#endif
	printk(KERN_NOTICE "Tasks: a: %x b: %x c: %x\n", usart_task_a, usart_task_b, usart_task);
// 	printk(KERN_NOTICE "shell=%x a=%x b=%x\n", usart_task, usart_task_a, usart_task_b);
// 	printk(KERN_NOTICE "a->next=%x b->next=%x\n", usart_task_a->next, usart_task_b->next);
	show_task_list();
// 	printk(KERN_NOTICE "Test string fmt: %20s %S\n", "asdf", ROSTR("Das ist ein String in RODATA"));
	
	init_cycles = read_hrpc();
	return 0;
}

initcall(init_usart_sched_timer_test);
