// SPDX-License-Identifier: GPL-2.0
// Copyright (c) 2008, 2018-2019 Rupert Eibauer. All rights reserved.

#include <linux/sched.h>
#include <linux/task.h>
#include <linux/printk.h>
#include <linux/delay.h>
#include <linux/init.h>

#include <asm/system.h>
#include <asm/io.h>
#include <asm/gpio.h>

//#define NO_WAIT
#ifdef NO_WAIT

#define wait() do { } while(0)

#else /* NO_WAIT */

#define wait() udelay(500000)

#endif /* NO_WAIT */

/* Simple priority inheritance test */
/* If priority inheritance is disabled, we will end up with
   t2 looping and t3 never getting mutex m1.
 */
/* The result should look like the following (when watched with simulavr-disp)
with CONFIG_MUTEX_PI:
 PORTA = 0x14
 PORTB = keeps incrementing
 PORTC = 0x34
without CONFIG_MUTEX_PI:
 PORTA = 0x13
 PORTB = keeps incrementing
 PORTC = 0x32
*/
/*
#define T1_ACTION(x) do{ gpiogroup_out(GPIO_A0, GPIO_A0_NBITS, x); }while(0)
#define T2_ACTION(x) do{ gpiogroup_out(GPIO_B0, GPIO_A0_NBITS, x); }while(0)
#define T3_ACTION(x) do{ gpiogroup_out(GPIO_C0, GPIO_A0_NBITS, x); }while(0)
*/


/* This is the 3-LED output variant of the test.
with CONFIG_MUTEX_PI:
 C0 = on
 C1 = blink
 C2 = on
without CONFIG_MUTEX_PI:
 C0 = on
 C1 = blink
 C2 = on
 If C0 or C2 are off, or C1 is not blinking, the test
 has failed.
*/

#ifdef CONFIG_MUTEX_PI
#define T1_ACTION(x) do{ gpiogroup_out(GPIO_C0, GPIO_C0_NBITS, (gpiogroup_in(GPIO_C0, GPIO_C0_NBITS) & ~1) | ((x == 0x14) ? 1 : 0)); wait(); }while(0)
#define T2_ACTION(x) do{ gpiogroup_out(GPIO_C0, GPIO_C0_NBITS, (gpiogroup_in(GPIO_C0, GPIO_C0_NBITS) & ~2) | ((x & 1) << 1));        wait(); }while(0)
#define T3_ACTION(x) do{ gpiogroup_out(GPIO_C0, GPIO_C0_NBITS, (gpiogroup_in(GPIO_C0, GPIO_C0_NBITS) & ~4) | ((x == 0x34) ? 4 : 0)); wait(); }while(0)
#else
#define T1_ACTION(x) do{ gpiogroup_out(GPIO_C0, GPIO_C0_NBITS, (gpiogroup_in(GPIO_C0, GPIO_C0_NBITS) & ~1) | ((x == 0x13) ? 1 : 0)); wait(); }while(0)
#define T2_ACTION(x) do{ gpiogroup_out(GPIO_C0, GPIO_C0_NBITS, (gpiogroup_in(GPIO_C0, GPIO_C0_NBITS) & ~2) | ((x & 1) << 1));        wait(); }while(0)
#define T3_ACTION(x) do{ gpiogroup_out(GPIO_C0, GPIO_C0_NBITS, (gpiogroup_in(GPIO_C0, GPIO_C0_NBITS) & ~4) | ((x == 0x32) ? 4 : 0)); wait(); }while(0)
#endif

struct completion s2;
struct completion s3;
struct mutex m1;

#ifdef CONFIG_STACK_SIZE_FIXED
struct task_struct t1;
struct task_struct t2;
struct task_struct t3;
#endif

/* The low priority process */
__kthread_func void f1(void *arg) {
	unsigned char d = 0x10;
	T1_ACTION(++d); // 0x11
	mutex_lock(&m1);
	T1_ACTION(++d); // 0x12
	complete(&s3);
	T1_ACTION(++d); // 0x13
	complete(&s2);
	T1_ACTION(++d); // 0x14
	mutex_unlock(&m1);
	T1_ACTION(++d); // 0x15
	for (;;);
}

/* The middle priority process */
__kthread_func void f2(void *arg) {
	unsigned char d = 0x20;
	T2_ACTION(++d);
	wait_for_completion(&s2);
	for (;;) {
		T2_ACTION(++d);
	}
}

/* The high priority process */
__kthread_func void f3(void *arg) {
	unsigned char d = 0x30;
	T3_ACTION(++d); // 0x31
	wait_for_completion(&s3); /* We wait until f1 has the mutex */
	T3_ACTION(++d); // 0x32
	mutex_lock(&m1);
	T3_ACTION(++d); // 0x33
	mutex_unlock(&m1);
	T3_ACTION(++d); // 0x34
	wait_for_completion(&s3); /* We wait again to let t1 continue */
	T3_ACTION(++d); // 0x35
}


extern void dump_memory(void);

static int init_mutex_test(void) {
	int i;
	gpiogroup_set_output_v(GPIO_C0, GPIO_C0_NBITS, 0x3f);
	wait();
	wait();
	wait();
	wait();
	gpiogroup_out(GPIO_C0, GPIO_C0_NBITS, 7);
	for (i = 0; i <= 7; i++) {
		wait();
		gpiogroup_out(GPIO_C0, GPIO_C0_NBITS, i);
	}
	wait();
	gpiogroup_out(GPIO_C0, GPIO_C0_NBITS, 0);
#ifdef CONFIG_STACK_SIZE_FIXED
	kernel_thread(&t1, f1, NULL, 1);
	kernel_thread(&t2, f2, NULL, 2);
	kernel_thread(&t3, f3, NULL, 3);
#else
	kernel_thread(128, f1, NULL, 1);
	kernel_thread(128, f2, NULL, 2);
	kernel_thread(128, f3, NULL, 3);
#endif
	gpiogroup_out(GPIO_C0, GPIO_C0_NBITS, 1);
	return 0;
}

initcall(init_mutex_test);
