// SPDX-License-Identifier: GPL-2.0
// Copyright (c) 2008, 2014, 2017-2019 Rupert Eibauer. All rights reserved.

#include <linux/types.h>
#include <linux/sched.h>
#include <linux/task.h>
#ifdef CONFIG_TICK_TIMER
#include <linux/timer.h>
#endif
#ifndef CONFIG_STACK_SIZE_FIXED
#include <linux/slab.h>
#endif
#include <linux/bug.h>
#include <linux/printk.h>

#include <asm/system.h>


extern void _taskSwitchTo(struct task_struct *newTask);

struct task_list {
	struct task_struct *first;
	struct task_struct *last;
};

#ifdef CONFIG_SCHED_RR

/* PRIO_IS_RR and HZ may also be adjusted by the user */
/* All odd priorities are Round_Robin */
#define PRIO_IS_RR(x)      (x & 1)

#define RR_TIME_SLICE(task)   (HZ / 100)

#else

#define PRIO_IS_RR(x) 0

#endif /* CONFIG_SCHED_RR */

#ifdef CONFIG_SCHED_OTHER

#define WEIGHT_TO_BONUS(b)   ((u16)(b) * (HZ / 100))

#endif /* CONFIG_SCHED_OTHER */


/* FIXME: Multiple waiters are required */
#define CONFIG_SEMA_MULTIPLE_WAITERS

#ifdef CONFIG_SEMA_MULTIPLE_WAITERS

#define WAIT_SET_NEXT(task, s)       task->next = s->waitTask
#define WAIT_UNSET_NEXT(task, s)     s->waitTask = task->next

#else /* CONFIG_SEMA_MULTIPLE_WAITERS */

#ifdef CONFIG_BUG

#define WAIT_SET_NEXT(task, s)       BUG_ON(s->waitTask)
#define WAIT_UNSET_NEXT(task, s)     s->waitTask = NULL

#else

#define WAIT_SET_NEXT(task, s)       do{}while(0)
#define WAIT_UNSET_NEXT(task, s)     do{}while(0)

#endif

#endif /* CONFIG_SEMA_MULTIPLE_WAITERS */

struct task_list prios[CONFIG_MAX_PRIOS];
#ifdef CONFIG_PER_PRIO_TIME_SLICES
u16 time_slices[CONFIG_MAX_PRIOS];
#endif
struct task_struct *current;

#if defined(CONFIG_WAKE_SEMA_FROM_IRQ) || defined(CONFIG_SCHED_OTHER)
bool need_resched;
bool in_schedule_loop;
#endif /* CONFIG_WAKE_SEMA_FROM_IRQ */

static
void _put_task(struct task_list *list, struct task_struct *task) {
	task->next = NULL;
	if (list->first == NULL) {
		list->first = task;
	} else {
		list->last->next = task;
	}
	list->last = task;
}

static
struct task_struct *_take_task(struct task_list *list) {
	struct task_struct *first;
	first = list->first;
	if (first)
		list->first = first->next;
	return first;
}

#if defined(CONFIG_SCHED_OTHER) || defined(CONFIG_SCHED_RR)

static void sched_timer_func(struct timer *timer);
struct timer sched_timer;
bool sched_timer_armed = 0;

static void disarm_sched_timer(struct task_struct *task) {
	if (sched_timer_armed) {
		_del_timer(&sched_timer);
	}
	/* FIXME: If we add a task to the list later, we will need to 
	   start the sched_timer. */
}

#endif /* CONFIG_SCHED_OTHER || CONFIG_SCHED_RR */

#ifdef CONFIG_SCHED_OTHER

struct task_struct *first_other;
/* Called to put the task onto the task list, after
   it has been running or when it is enqueued the first
   time.
   If it is run the first time, or if all tasks have been running, 
   it will be at the beginning of the first_other list, and the time
   slice size of the tasks will be recalculated.
 */
static
void _taskEnqueue(struct task_struct *task, int prio) {
	struct task_list *list = &prios[prio];

	if (prio == 0) {
		struct task_struct * *tiP;
		struct task_struct *ti;
		/* This time-consuming work is done now. */
		/* Linux uses a rbtree for this, but this is overkill for AVR */
		for (tiP = &list->first, ti = *tiP; ti != NULL; tiP = &ti->next, ti = *tiP) {
			if (ti->entity.bonus < task->entity.bonus)
				break;
		}
		task->next = ti;
		if (!ti)
			list->last = task;
		*tiP = task;
		return;
	}
	_put_task(list, task);
}

static inline 
void update_curr(struct task_struct *task) {
	/* Do not use prio! The task could be SCHED_OTHER, but have inherited
	   a higher prio.
	  */
	if (task->entity.weight) {
		u16 runTime = jiffies - task->entity.exec_start;
		task->entity.bonus -= runTime;
	}
	disarm_sched_timer(task);
}

#else /* CONFIG_SCHED_OTHER */

#define _taskEnqueue(task, prio) _put_task(&prios[prio], task)

#ifdef CONFIG_SCHED_RR
#define update_curr(task) disarm_sched_timer(task)
#else
#define update_curr(task) do{}while(0)
#endif

#endif /* CONFIG_SCHED_OTHER */

#ifdef CONFIG_TICK_TIMER

/* Switch to a task of the same priority */
static void _taskSelect(struct task_struct *task) {
	/* taskSelect is only called when the sched_timer is already expired. 
	   FIXME: Sure? If the task has suspended by a semaphore or mutex, or by an interrupt handler
	          freeing a higher-prio task, it could well be that the timer has not yet expired.
	          We should check the timer if it has expired. */
	/* If there is no other task on the same priority,
	   we don't need to schedule a context switch */
	BUG_ON(!task);
#ifdef CONFIG_SCHED_OTHER
	if (task->entity.weight && task->entity.bonus <= 0) { /* Distribute new bonii */
		struct task_struct *t;
		for (t = first_other; t; t = t->entity.next_other) {
			/* The task can keep 1/2 of its old unused bonus. 
				If we give it more, t->bonus could overflow. 
				However, it might be better if it could keep 3/4 or more.
				*/
			s16 old_bonus, new_bonus;
			old_bonus = t->entity.bonus;
			new_bonus = old_bonus + WEIGHT_TO_BONUS(t->entity.weight);
			if (new_bonus < old_bonus)
				new_bonus = 32767;
			t->entity.bonus = new_bonus;
		}
	}
#endif
#if defined(CONFIG_SCHED_RR) || defined(CONFIG_SCHED_OTHER)
	if (!prios[task->entity.prio].first) {
		return;
	}
#endif
#ifdef CONFIG_SCHED_RR
	if (PRIO_IS_RR(task->entity.prio)) {
		sched_timer.expires = jiffies + RR_TIME_SLICE(task);
		sched_timer.function = sched_timer_func;
		_add_timer(&sched_timer);
		sched_timer_armed = 1;
		return;
	}
#endif /* CONFIG_SCHED_RR */
#ifdef CONFIG_SCHED_OTHER
	if (task->entity.weight) {
		task->entity.exec_start = jiffies;
		if (task->entity.prio == 0) {
			/* If the task did not inherit a higher prio, we schedule it for removal. */
			sched_timer.expires = jiffies + task->entity.bonus;
			sched_timer.function = sched_timer_func;
			_add_timer(&sched_timer);
			sched_timer_armed = 1;
		}
	}
#endif /* CONFIG_SCHED_OTHER */
}

static void sched_timer_func(struct timer *timer) {

	sched_timer_armed = 0;
	need_resched = 1;
}

#else /* CONFIG_TICK_TIMER */

#define _taskSelect(task) do{}while(0)

#endif /* CONFIG_TICK_TIMER */


#if defined(CONFIG_STACK_SIZE_REPORTING) || defined(CONFIG_STACK_CHECK)

#define CONFIG_THREAD_LIST
#ifdef CONFIG_THREAD_LIST
#define MAX_THREADS 5
static struct task_struct *threads[MAX_THREADS];
static u8 num_threads;
#endif

#endif /* CONFIG_STACK_SIZE_REPORTING || CONFIG_STACK_CHECK */

#ifdef CONFIG_STACK_CHECK

void _check_thread_stack(struct task_struct *task, void *ret, u16 line) {
	int a;
	uintptr_t sp;
	stack_size_t stack_size;

	if (task == NULL)
		return;

#ifndef CONFIG_STACK_SIZE_FIXED
	stack_size = task->stack_size;
#else
	stack_size = CONFIG_STACK_SIZE;
#endif

	if (task == current) {
		sp = in_16(SP);
	} else {
#ifdef CONFIG_LARGE_SP
		sp = (uintptr_t)task->state.L_SP;
#else
		sp = (uintptr_t)&task->stack[task->state.sp];
#endif
	}
	if ((uintptr_t)&task->stack[stack_size] <= sp || sp <= (uintptr_t)task->stack - 10) {
		goto stack_panic;
	}
#ifdef CONFIG_THREAD_LIST
	for (a = 0; a < num_threads; a++) {
		if (threads[a] == task) {
			break;
		}
	}
	if (a == num_threads) {
		printk(KERN_EMERG "Unknown thread %04x ret=%04x line=%d\n", (u16)task, ret, line);
		panic("Unknown thread");
	}
#endif

#if CONFIG_STACK_GUARD > 0
	for (a = 0; a < CONFIG_STACK_GUARD; a++) {
		if (task->stack_guard1[a] != '1') {
			goto stack_panic;
		}
		if (task->stack[stack_size + a] != '2') {
			goto stack_panic;
		}
	}
#endif
	return;
stack_panic:
	printk(KERN_CRIT "stack panic: PC=%04x task=%04x current=%04x SP=%04x line %d!\n", (u16)__builtin_return_address(0), task, current, sp, line);
	panic("stack panic");
}

#endif /* CONFIG_STACK_CHECK */

#ifdef CONFIG_STACK_SIZE_REPORTING

void report_stack_usage(void) {
	u16 a, i;
	struct task_struct *task;

	for (i = 0; i < num_threads; i++) {
		stack_size_t stack_size;
		task = threads[i];
#ifdef CONFIG_STACK_SIZE_FIXED
		stack_size = CONFIG_STACK_SIZE;
#else
		stack_size = task->stack_size;;
#endif
		for (a = 0; a < stack_size - 2; a++)
			if (task->stack[a] != 0xaa) {
#ifdef CONFIG_PRINTK
				printk(KERN_NOTICE "%04x:%d\n", task, stack_size - a);
#else
				extern void report_stack_size(uintptr_t task, uintptr_t stack_usage);
				report_stack_size((uintptr_t)task, stack_size - a);
#endif
				break;
			}
	}
}

#endif /* CONFIG_STACK_SIZE_REPORTING */

#ifndef CONFIG_STACK_SIZE_FIXED
struct task_struct *kernel_thread(stack_size_t stack_size, void (*f)(void*), void *arg, signed char prio)
#else
struct task_struct *kernel_thread(struct task_struct *task, void (*f)(void*), void *arg, signed char prio)
#endif
{
	irqflags_t flags;
#ifndef CONFIG_STACK_SIZE_FIXED
	struct task_struct *task;

	if (stack_size > CONFIG_STACK_SIZE_MAX) {
		return NULL;
	}
	task = kmalloc(sizeof(struct task_struct) + stack_size + CONFIG_STACK_GUARD, GFP_KTHREAD);
	if (!task)
		return task;
#ifdef CONFIG_STACK_CHECK
	task->stack_size = stack_size;
#endif
#else
	stack_size_t stack_size = CONFIG_STACK_SIZE;
#endif

	setup_task(task, f, arg, stack_size);

#ifdef CONFIG_STACK_SIZE_REPORTING
	printk(KERN_INFO "kernel_thread(%04x, %d)\n", task, prio);
#endif

#ifdef CONFIG_SCHED_OTHER
	if (prio > 0) {
		task->entity.prio = prio;
	} else {
		u8 weight;
		task->entity.prio = 0;
		weight = 1 << ((-prio + 1) >> 1);
		if ((prio & 1) == 0)
			weight += weight >> 1;
		task->entity.weight = weight;
		task->entity.bonus = WEIGHT_TO_BONUS(weight);
	}
#else
	task->entity.prio = prio;
#endif
	
	local_irq_save(flags);
#ifdef CONFIG_THREAD_LIST
	if (num_threads == MAX_THREADS) {
		printk(KERN_WARNING "Thread limit reached\n");
	} else {
		threads[num_threads] = task;
		num_threads++;
	}
#endif

#ifdef CONFIG_SCHED_OTHER
	if (prio <= 0) {
		/* Insert at the beginning to make a re-calc of boni */
		task->entity.next_other = first_other;
		first_other = task;
		prio = 0;
	}
#endif
	_taskEnqueue(task, prio);
	local_irq_restore(flags);
	return task;
}

static void _schedule(void)
{
	signed char i;
	struct task_struct *cur = current;
	struct task_struct *t;

	check_thread_stack(cur);
	for (;;) {
		for (i = CONFIG_MAX_PRIOS - 1; i >= 0; i--) {
			t = _take_task(&prios[i]);
			if (t) {
				goto found;
			}
		}

#if !defined(CONFIG_IRQ)
		panic("no next task and no interrupts");
#endif
		/* No task found. Set processor to sleep and repeat when woken up. */
		/* The job of the idle thread is done _just_ here. */
		/* We enable interrupt for the case that the system will be restarted by
		   a future interrupt */
		in_schedule_loop = 1;
		safe_halt();          // safe_halt enables interrupts.
		local_irq_disable();
		in_schedule_loop = 0;
	}
found:
	need_resched = 0;
	check_thread_stack(t);
	check_thread_stack(cur);

	_taskSelect(t);
	if (t != cur) {
		_taskSwitchTo(t);
	}
}

void kill_current(void)
{
	local_irq_disable();
	_schedule();
	__builtin_unreachable();
}

#ifdef CONFIG_USE_SEMA
void complete(struct completion *s)
{
	struct task_struct *t;
	signed char done;

	local_irq_disable();
	done = ++s->done;
	if (done == 0) {
		t = s->waitTask;
		BUG_ON(t == NULL);
		WAIT_UNSET_NEXT(t, s);
		check_thread_stack(t);
		BUG_ON(current == NULL);
		check_thread_stack(current);
		BUG_ON(current == t);

		if (t->entity.prio > current->entity.prio) {
			/* Switch to the task */
			_taskEnqueue(current, current->entity.prio);
			update_curr(current);
			_taskSelect(t);
			_taskSwitchTo(t);
		} else {
			/* Add task to the runnable list */
			_taskEnqueue(t, t->entity.prio);
		}
	}
	local_irq_enable();
}

#ifdef CONFIG_WAKE_SEMA_FROM_IRQ

void complete_from_irq(struct completion *s)
{
	struct task_struct *t;
	s8 done;

	done = ++s->done;
	if (done == 0) {
		t = s->waitTask;
		WAIT_UNSET_NEXT(t, s);
		_taskEnqueue(t, t->entity.prio);
		if (t != current) {
			need_resched = 1;
		}
	}
}

void resched_after_irq(void) {

// 	check_thread_stack(current);
	if (need_resched && !in_schedule_loop) {
		_taskEnqueue(current, current->entity.prio);
		update_curr(current);
		_schedule();
	}
}

#endif /* CONFIG_WAKE_SEMA_FROM_IRQ */

void wait_for_completion(struct completion *s)
{
	signed char done;

	BUG_ON(irqs_disabled());
	BUG_ON(current == NULL);
	check_thread_stack(current);
	local_irq_disable();
	done = --s->done;
	if (done < 0) {
		WAIT_SET_NEXT(current, s);
		s->waitTask = current;
		update_curr(current);
		_schedule();
	}
	local_irq_enable();
}
#endif /* CONFIG_USE_SEMA */


#if defined(CONFIG_USE_CHANGE_PRIO) || defined(CONFIG_MUTEX_PI)
static void _taskChangePrio(struct task_struct *task, u8 prio) {
	struct task_struct *tP;
	struct task_struct *tPPrev;
	struct task_list *task_list = &prios[task->entity.prio];

	tPPrev = NULL;
	task->entity.prio = prio;
#ifdef CONFIG_MUTEX_PI
	if (task->waitMutex != NULL) {
		_taskChangePrio(task->waitMutex->owner, prio);
		return;
	}
#endif
	for (tP = task_list->first; tP != NULL; tP = tP->next) {
		/* FIXME: We could optimize this loop away if we had 
		   - a doubly-linked list here 
		   - a task state saved in the task
		 */
		if (tP == task) {
			/* Task found on list. Remove it and put it
			   onto the list of its new RQ.
			 */
			if (tPPrev != NULL) {
				tPPrev->next = tP->next;
			}
			if (task == task_list->last) {
				task_list->last = tPPrev;
			} else {
				task_list->first = tP->next;
			}
			task->entity.prio = prio;
			_taskEnqueue(task, prio);
			return;
		}
		tPPrev = tP;
	}
}
#endif /* defined(TASKS_USE_MUTEX_PI) || defined(CONFIG_MUTEX_PI) */

#ifdef CONFIG_USE_CHANGE_PRIO
void taskChangePrio(struct task_struct *task, u8 prio) {

	BUG_ON(current == NULL);
	if (task == current || task == NULL) {
		current->entity.prio = prio;
		return;
	}
	local_irq_disable();
	_taskChangePrio(task, prio);
	local_irq_enable();
}
#endif /* CONFIG_USE_CHANGE_PRIO */

/* Optional scheduler features */
#ifdef CONFIG_USE_MUTEX

void mutex_lock(struct mutex *m)
{
	struct task_struct *cur = current;

	local_irq_disable();
	if (m->owner == NULL) {
		m->owner = cur;
#ifdef CONFIG_MUTEX_PI
		m->origPrio = cur->entity.prio;
#endif /* CONFIG_MUTEX_PI */
	} else {
		/* Enqueue the current task after the last same-prio or 
		   before the first lower-prio task  in m->waiters */
		struct task_struct * *tiP;
		struct task_struct *ti;
		for (tiP = &m->waiters, ti = *tiP; ti != NULL; tiP = &ti->next, ti = *tiP) {
			if (ti->entity.prio < cur->entity.prio)
				break;
		}
		BUG_ON(cur == NULL);
		cur->next = ti;
		*tiP = cur;
#ifdef CONFIG_MUTEX_PI
		if (cur->entity.prio > m->owner->entity.prio) {
			_taskChangePrio(m->owner, cur->entity.prio);
		}
		cur->waitMutex = m;
#endif /* CONFIG_MUTEX_PI */
		update_curr(cur);
		_schedule();
	}
	local_irq_enable();
}

void mutex_unlock(struct mutex *m)
{
	struct task_struct *t;

	local_irq_disable();
	BUG_ON(current == NULL);
#ifdef CONFIG_MUTEX_PI
	current->entity.prio = m->origPrio;
#endif /* CONFIG_MUTEX_PI */

	/* get the highest prio waiter and make it the new Owner */
	t = m->waiters;
	m->owner = t;

	if (t == NULL) {
		goto out;
	}

	m->waiters = t->next;
#ifdef CONFIG_MUTEX_PI
	m->origPrio = t->entity.prio;
	t->waitMutex = NULL;
#endif /* CONFIG_MUTEX_PI */
	if (t->entity.prio > current->entity.prio) {
		_taskEnqueue(current, current->entity.prio);
		update_curr(current);
		BUG_ON(t == current);
		check_thread_stack(t);
		check_thread_stack(current);
		_taskSelect(t);
		_taskSwitchTo(t);
	} else {
		_taskEnqueue(t, t->entity.prio);
	}

out:
	local_irq_enable();
}

#endif /* CONFIG_USE_MUTEX */
