// SPDX-License-Identifier: GPL-2.0
// Copyright (c) 2010, 2017-2019 Rupert Eibauer. All rights reserved.

#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/timer.h>
#include <linux/cdev.h>

#include <asm/system.h>
#include <asm/rodata.h>
#include <asm/gpio.h>

#ifndef MORSE_GPIO

// The new API will provide MORSE_GPIO defined from the Makefile

#define GPIO_DEF_(a, b) GPIO_##a##b

#define GPIO_DEF(a, b) GPIO_DEF_(a, b)

#ifdef CONFIG_SERIAL_MORSE_PORTA
#define MORSE_GPIO GPIO_DEF(A, CONFIG_SERIAL_MORSE_BIT)
#elif defined(CONFIG_SERIAL_MORSE_PORTB)
#define MORSE_GPIO GPIO_DEF(B, CONFIG_SERIAL_MORSE_BIT)
#elif defined(CONFIG_SERIAL_MORSE_PORTC)
#define MORSE_GPIO GPIO_DEF(C, CONFIG_SERIAL_MORSE_BIT)
#elif defined(CONFIG_SERIAL_MORSE_PORTD)
#define MORSE_GPIO GPIO_DEF(D, CONFIG_SERIAL_MORSE_BIT)
#elif defined(CONFIG_SERIAL_MORSE_PORTE)
#define MORSE_GPIO GPIO_DEF(E, CONFIG_SERIAL_MORSE_BIT)
#elif defined(CONFIG_SERIAL_MORSE_PORTF)
#define MORSE_GPIO GPIO_DEF(F, CONFIG_SERIAL_MORSE_BIT)
#else
#error define a port
#endif

#endif

union morsechar {
	struct {
		u8 length;
		u8 code;
	};
	u16 combined;
};

#define MC(...) MC_(__VA_ARGS__, 0, 0, 0, 0, 0, 0, 0, 8, 7, 6, 5, 4, 3, 2, 1, 0, 0, 0, 0, 0, 0, 0, 0)
#define MC_(x0, x1, x2, x3, x4, x5, x6, x7, x8, x9, x10, x11, x12, x13, x14, count, ...) MC__(count, x0, x1, x2, x3, x4, x5, x6, x7)
#define MC__(l, x0, x1, x2, x3, x4, x5, x6, x7) { { \
	.length = l, \
	.code = x0 | \
	       (x1 << 1) | \
	       (x2 << 2) | \
	       (x3 << 3) | \
	       (x4 << 4) | \
	       (x5 << 5) | \
	       (x6 << 6) | \
	       (x7 << 7), \
} }

#define DIT 0
#define DAH 1

static union morsechar __rodata morse_alphabet[] = {
	['\b'] = MC(DIT, DIT, DIT, DIT, DIT, DIT, DIT, DIT),
	['\n'] = MC(DIT, DAH, DIT, DAH, DIT), // AR (Spruchende)
	['!']  = MC(DAH, DIT, DAH, DIT, DAH, DAH), // KW digraph (Heathkit)
// 	['!']  = MC(DAH, DAH, DAH, DIT), // MN digraph (American Morse landline code)
	['"']  = MC(DIT, DAH, DIT, DIT, DAH, DIT),
	['#']  = MC(DIT, DIT, DIT, DIT, DIT, DIT, DIT), // Non-standard HS digraph
	['$']  = MC(DIT, DIT, DIT, DAH, DIT, DIT, DAH), // SX digraph (Phillips Code)
	['%']  = MC(DIT, DAH, DAH, DIT, DAH, DIT, DAH, DIT), // Non-standard PC digraph
	['&']  = MC(DIT, DAH, DIT, DIT, DIT), // AS digraph, Prosign for wait
	['\''] = MC(DIT, DAH, DAH, DAH, DAH, DIT),
	['(']  = MC(DAH, DIT, DAH, DAH, DIT),
	[')']  = MC(DAH, DIT, DAH, DAH, DIT, DAH),
	['*']  = MC(DIT, DAH, DAH, DIT, DAH), // Non-standard AK digraph
	['+']  = MC(DIT, DAH, DIT, DAH, DIT),
	[',']  = MC(DAH, DAH, DIT, DIT, DAH, DAH),
	['-']  = MC(DAH, DIT, DIT, DIT, DIT, DAH),
	['.']  = MC(DIT, DAH, DIT, DAH, DIT, DAH),
	['/']  = MC(DAH, DIT, DIT, DAH, DIT),
	['0']  = MC(DAH, DAH, DAH, DAH, DAH),
	['1']  = MC(DIT, DAH, DAH, DAH, DAH),
	['2']  = MC(DIT, DIT, DAH, DAH, DAH),
	['3']  = MC(DIT, DIT, DIT, DAH, DAH),
	['4']  = MC(DIT, DIT, DIT, DIT, DAH),
	['5']  = MC(DIT, DIT, DIT, DIT, DIT),
	['6']  = MC(DAH, DIT, DIT, DIT, DIT),
	['7']  = MC(DAH, DAH, DIT, DIT, DIT),
	['8']  = MC(DAH, DAH, DAH, DIT, DIT),
	['9']  = MC(DAH, DAH, DAH, DAH, DIT),
	[':']  = MC(DAH, DAH, DAH, DIT, DIT, DIT),
	[';']  = MC(DAH, DIT, DAH, DIT, DAH, DIT),
	['<']  = MC(DIT, DAH, DIT, DIT, DAH), // Non-standard LT digraph
	['=']  = MC(DAH, DIT, DIT, DIT, DAH),
	['>']  = MC(DAH, DAH, DIT, DAH, DIT, DIT, DIT), // Non-standard GTH trigraph, shortened to 7 bit
	['?']  = MC(DIT, DIT, DAH, DAH, DIT, DIT),
	['@']  = MC(DIT, DAH, DAH, DIT, DAH, DIT), // AC digraph
	['A']  = MC(DIT, DAH),
	['B']  = MC(DAH, DIT, DIT, DIT),
	['C']  = MC(DAH, DIT, DAH, DIT),
	['D']  = MC(DAH, DIT, DIT),
	['E']  = MC(DIT),
	['F']  = MC(DIT, DIT, DAH, DIT),
	['G']  = MC(DAH, DAH, DIT),
	['H']  = MC(DIT, DIT, DIT, DIT),
	['I']  = MC(DIT, DIT),
	['J']  = MC(DIT, DAH, DAH, DAH),
	['K']  = MC(DAH, DIT, DAH),
	['L']  = MC(DIT, DAH, DIT, DIT),
	['M']  = MC(DAH, DAH),
	['N']  = MC(DAH, DIT),
	['O']  = MC(DAH, DAH, DAH),
	['P']  = MC(DIT, DAH, DAH, DIT),
	['Q']  = MC(DAH, DAH, DIT, DAH),
	['R']  = MC(DIT, DAH, DIT),
	['S']  = MC(DIT, DIT, DIT),
	['T']  = MC(DAH),
	['U']  = MC(DIT, DIT, DAH),
	['V']  = MC(DIT, DIT, DIT, DAH),
	['W']  = MC(DIT, DAH, DAH),
	['X']  = MC(DAH, DIT, DIT, DAH),
	['Y']  = MC(DAH, DIT, DAH, DAH),
	['Z']  = MC(DAH, DAH, DIT, DIT),
// 	['[']  = MC(),
// 	['\\'] = MC(),
// 	[']']  = MC(),
// 	['^']  = MC(),
	['_']  = MC(DIT, DIT, DAH, DAH, DIT, DAH),
};

static const jiffies_t dit_length = HZ / CONFIG_SERIAL_MORSE_SPEED;

#if CONFIG_SERIAL_MORSE_SOUND_FREQ > 0
static struct timer ton_timer;
static void ton_callback(struct timer *timer) {
	gpio_toggle(MORSE_GPIO);
	timer->expires += HZ / CONFIG_SERIAL_MORSE_SOUND_FREQ;
	_add_timer(timer);
}
static struct timer ton_timer = {
	.function = ton_callback,
};
static inline void morse_on(void) {
	ton_timer.expires = jiffies + HZ / CONFIG_SERIAL_MORSE_SOUND_FREQ;
	_add_timer(&ton_timer);
}
static inline void morse_off(void) {
	_del_timer(&ton_timer);
	gpio_out(MORSE_GPIO, 0);
}
#else
static inline void morse_on(void) {
	gpio_out(MORSE_GPIO, 1);
}
static inline void morse_off(void) {
	gpio_out(MORSE_GPIO, 0);
}
#endif

#ifdef CONFIG_MORSE_CONSOLE
struct console_dev {
	void *putc(u8 ch);
	
};

struct console_dev {
	struct console_dev base;
	u8 nbits;
	u8 bits;
	
} dev;

#endif

static s16 get_next_char(void);

static struct timer morse_timer;

static jiffies_t next_pause;


static void morse_callback(struct timer *timer);
static void pause_callback(struct timer *timer) {
	morse_off();
	timer->function = morse_callback;
	timer->expires += next_pause;
	_add_timer(timer);
}
static u8 morse_running = 0;
static void morse_callback(struct timer *timer) {
	static union morsechar current_char;

	if (current_char.length == 0) {
		s16 next_char;
		u8 ch;

		next_char = get_next_char();
		if (next_char == -1) {
			morse_running = 0;
//			morse_off();
			return;
		}
		ch = next_char;
		if (ch == ' ') {
			timer->expires += dit_length * 7;
			_add_timer(timer);
			return;
		}
		if (ch >= 'a' && ch <= 'z')
			ch -= 'a' - 'A'; // Lowercase letters are encoded as upper-case.

		if (ch < ARRAY_SIZE(morse_alphabet))
			current_char.combined = morse_alphabet[ch].combined;
		if (!current_char.length) {
			/* Unknown characters are encoded 8-Bit. This is safe, because 
			   the only other 8-bit code is HH, which has the same encoding 
			   as the nul-character, which is filtered out anyway */
			current_char.length = 8;
			current_char.code = ch;
		}
	}
	if (current_char.code & 1) { // dat is 3 times longer
		timer->expires += dit_length * 3;
	} else {
		timer->expires += dit_length;
	}
	current_char.length--;
	current_char.code >>= 1;
	if (current_char.length == 0) {
		next_pause = dit_length * 3;
	} else {
		next_pause = dit_length;
	}
	morse_on();
	timer->function = pause_callback;
	_add_timer(timer);
}
static struct timer morse_timer = {
	.function = morse_callback,
	.expires  = HZ + 1 + HZ / 880,
};

/* buffered console */
#define RB_SHIFT 8
#include "rbtmpl.h"

static RB_T morse_rb;
static s16 get_next_char(void) {
	return RB_GET_CHAR(morse_rb);
}

static void _morse_putc(u8 ch) {
	if (!ch) /* Ignore NULL characters. */
		return;

	if (ch >= 'a' && ch <= 'z')
		ch -= 'a' - 'A';
	RB_PUT_CHAR(morse_rb, ch);
	if (!morse_running) {
		morse_running = 1;
		morse_timer.expires = jiffies + dit_length * 7;
		add_timer(&morse_timer);
	}
}

#ifdef CONFIG_MORSE_CONSOLE

static void morse_cdev_putc(struct ocdevbase *dev, u8 ch) {
	return morse_putc_(ch);
}

struct ocdev kernel_console = {
	.out.putc_func = morse_cdev_putc,
};
#warning needs testing

#else

void morse_putc(u8 ch) {
	return _morse_putc(ch);
}

#endif


#ifdef CONFIG_TEST_MORSE
void morse_init(void) {
	gpio_set_output(MORSE_GPIO);
}

extern void morse_puts_p(__rodata char *progmem_s);
#define morse_puts_P(__s) morse_puts_p(ROSTR(__s))

void morse_puts_p(__rodata char *str)
{
	char ch;
	while ((ch = *str))
	{
		morse_putc(ch);
		str++;
	}
}
void morse_puts(char *ch) {
	while (*ch) {
		morse_putc(*ch);
		ch++;
	}
}

int main(void) {

	local_irq_enable();

	morse_init();
	morse_puts_P("0 1 2 3 4 5 6 7 8 9  ");
	morse_puts_P("Hallo, Das ist Morsen mit rupsched! Viel Spass!  ");
	morse_puts_P("Hallo, Welt!");
	morse_puts_P("A B C D E F G H I J K L M N O P Q R S T U V W X Y Z");
	for(;;);
	return 0;
}
#endif
