// SPDX-License-Identifier: GPL-2.0
// Copyright (c) 2010, 2014, 2017-2019 Rupert Eibauer. All rights reserved.

#include <linux/types.h>
#include <linux/sched.h>
#include <linux/delay.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/cdev.h>
#include <linux/compiler.h>

#include <rupsched/avr232.h>

#include <asm/system.h>
#include <asm/io.h>
#include <asm/irqvec.h>

/* Default behavior */
#define BLOCKING_TX
#define BLOCKING_RX

/* Enable this if you are sure to never call avr232_getc from main, irq or irq-disabled context */
/* Otherwise, you can get a non-blocking RX by simply doing disable_irq() before avr232_getc() */
#define RX_NOIRQCHECK
/* Enable this if you are sure to never call avr232_putc from main, irq or irq-disabled context */
/* Otherwise, you can get non-blocking behaviour by doing disable_irq() before avr232_putc() */
// #define TX_NOIRQCHECK  


/* Warning: This causes extra interrupts */
// #undef CONFIG_OPTIMIZE_SIZE



#define UBRR_VALUE (CONFIG_CPUFREQ / 16 / CONFIG_SERIAL_AVR_BAUD - 1)

#define REAL_BAUD (CONFIG_CPUFREQ / 16 / (UBRR_VALUE + 1))

#if (REAL_BAUD > CONFIG_SERIAL_AVR_BAUD * 101 / 100)
#error Baud rate out of spec (too high)
#endif

#if (REAL_BAUD < CONFIG_SERIAL_AVR_BAUD * 99 / 100)
#error Baud rate of spec (too low)
#endif

// FIXME: get rid of CONFIG_AVR_UART_NEW
#ifdef UDR0

#ifndef CONFIG_AVR_UART_NEW
#define CONFIG_AVR_UART_NEW
#endif

#define UDR          UDR0

#define UBRRH        UBRR0H
#define UBRRL        UBRR0L

#define UCSRA        UCSR0A
#define UCSRA_RXC    UCSR0A_RXC0
#define UCSRA_TXC    UCSR0A_TXC0

#define UCSRA_FE     UCSR0A_FE0
#define UCSRA_DOR    UCSR0A_DOR0
#define UCSRA_UPE    UCSR0A_UPE0

#define UCSRB UCSR0B
#define UCSRB_RXCIE  UCSR0B_RXCIE0
#define UCSRB_TXCIE  UCSR0B_TXCIE0
#define UCSRB_UDRIE  UCSR0B_UDRIE0
#define UCSRB_RXEN   UCSR0B_RXEN0
#define UCSRB_TXEN   UCSR0B_TXEN0
#define UCSRB_UCSZ2  UCSR0B_UCSZ02
#define UCSRB_RXB8   UCSR0B_RXB80
#define UCSRB_TXB8   UCSR0B_TXB80


#define UCSRC            UCSR0C
#define UCSRC_USBS       UCSR0C_USBS0
#define UCSRC_UCSZ_MASK  UCSR0C_UCSZ0_MASK

#endif /* UDR0 */

#ifndef IRQ_USART_RX

#ifdef IRQ_USART_RXC
#define IRQ_USART_RX IRQ_USART_RXC
#elif defined IRQ_USART0_RX
#define IRQ_USART_RX IRQ_USART0_RX
#else
#error Dont know how to find IRQ_USART_RX
#endif

#endif /* IRQ_USART_RX */

#ifndef IRQ_USART_UDRE

#ifdef IRQ_USART0_UDRE
#define IRQ_USART_UDRE IRQ_USART0_UDRE
#else
#error Dont know how to find IRQ_USART_UDRE
#endif

#endif /* IRQ_USART_UDRE */

static inline void tx(char data) {
	out(UDR, data);
}

static inline int tx_buffer_full(void) {
	return !(in(UCSRA) & UCSRB_UDRIE);
}

static __maybe_unused void noirq_putc(char data) {
	/* Wait for empty transmit buffer */
	while (tx_buffer_full())
		{ }
	tx(data);
}

#ifdef CONFIG_SERIAL_AVR_TX

#ifdef CONFIG_SERIAL_AVR_TX_IRQMODE

#define RB_SHIFT CONFIG_SERIAL_AVR_TX_FIFO_SIZE
#include "rbtmpl.h"
RB_T tx_rb;
u8 tx_active;

static inline void disable_tx_irq(void) {
	out(UCSRB, in(UCSRB) & ~UCSRB_UDRIE);
}

static inline void enable_tx_irq(void) {
	out(UCSRB, in(UCSRB) | UCSRB_UDRIE);
}

#ifdef CONFIG_WAKE_SEMA_FROM_IRQ

struct completion tx_completion;

int avr232_putc(char data) {

#ifndef TX_NOIRQCHECK
	if (irqs_disabled()) {
		noirq_putc(data);
		return data;
	}
#endif

#ifdef BLOCKING_TX
	wait_for_completion(&tx_completion);
#else
	if (!try_wait_for_completion(&tx_completion))
		return -1;
#endif
	local_irq_disable(); 
// 	mutex_lock(&tx_lock); // alternative locking could improve interrupt latency.
// 	disable_tx_irq();

	if (tx_active) {
		RB_PUT_CHAR_NOCHECK(tx_rb, data);
		local_irq_enable();
// 		enable_tx_irq();
// 		mutex_unlock(&tx_lock);
		return data;
	}
#ifndef CONFIG_OPTIMIZE_SIZE
	if (!tx_buffer_full()) { /* If possible, send directly */
		tx(data);
		local_irq_enable();
// 		mutex_unlock(&tx_lock);
		complete(&tx_completion);
		return data;
	}
#endif
	/* Sender still active. give it to the ring buffer and start the sender */
	RB_PUT_CHAR_NOCHECK(tx_rb, data); /* Cannot be full because tx is inactive. */
	tx_active = 1;
	local_irq_enable();
	enable_tx_irq();
// 	mutex_unlock(&tx_lock);
	return data;
}

static void avr232_txirq(u8 irq) {
	int ch;

	ch = RB_GET_CHAR_NOCHECK(tx_rb);
	complete_from_irq(&tx_completion);
	tx(ch);
	if (tx_completion.done == (1 << RB_SHIFT) - 1) {
		tx_active = 0;
		disable_tx_irq();
	}
}

#else /* CONFIG_WAKE_SEMA_FROM_IRQ */

int avr232_putc(char data) {

	local_irq_disable();

	if (tx_active) {
#ifdef BLOCKING_TX
		while (RB_PUT_CHAR(tx_rb, data)) {
			local_irq_enable();
			udelay(10000000 / CONFIG_SERIAL_AVR_BAUD);
			local_irq_disable();
		}
#else
               if (RB_PUT_CHAR(tx_rb, data)) {
                       local_irq_enable();
                       return -1;
//                     return EOF;
               }
#endif
		local_irq_enable();
		return data;
	}
#ifndef CONFIG_OPTIMIZE_SIZE
	if (!tx_buffer_full()) { /* If possible, send directly */
		tx(data);
		local_irq_enable();
		return data;
	}
#endif
	/* Sender still active. give it to the ring buffer and start the sender */
	RB_PUT_CHAR_NOCHECK(tx_rb, data); /* Cannot normally be full if we just noticed that tx is inactive. */
	tx_active = 1;
	local_irq_enable();
	enable_tx_irq();
	return data;
}

static void avr232_txirq(u8 irq) {
	int ch;

	ch = RB_GET_CHAR_NOCHECK(tx_rb);

	tx(ch);

	if (RB_EMPTY(tx_rb)) {
		tx_active = 0;
		disable_tx_irq();
	}
}

#endif /* CONFIG_WAKE_SEMA_FROM_IRQ */

REQUEST_IRQ(IRQ_USART_UDRE, avr232_txirq);

static void avr232_tx_init(void) {
#ifdef CONFIG_WAKE_SEMA_FROM_IRQ
	/* FIXME: Somehow it works with 1 byte less! Find out why. */
	tx_completion.done = (1 << RB_SHIFT) - 1;
#endif
}

#else /* CONFIG_SERIAL_AVR_TX_IRQMODE */

int avr232_putc(char data) {
	/* Wait for empty transmit buffer */
	irqflags_t flags;

	local_irq_save(flags);
	noirq_putc(data);
	local_irq_restore(flags);
	return data;
}

static void avr232_tx_init(void) {
//	enable_tx_irq();
}

#endif /* CONFIG_SERIAL_AVR_TX_IRQMODE */

#endif /* CONFIG_SERIAL_AVR_TX */

#ifdef CONFIG_SERIAL_AVR_RX

static inline u8 rx(void) {
	return in(UDR);
}

static inline int rx_error(void) {
	return in(UCSRA) & (UCSRA_FE | UCSRA_DOR | UCSRA_UPE);
}

static inline int rx_buffer_empty(void) {
	return !(in(UCSRA) & UCSRA_RXC);
}

static inline s16 getc_irqsave(void) {
	irqflags_t flags;
	s16 ch = -1;

	local_irq_save(flags);

	if (!rx_buffer_empty())
		ch = rx();

	local_irq_restore(flags);
	return ch;
}

#ifdef CONFIG_SERIAL_AVR_RX_IRQMODE

#undef RB_SHIFT
#define RB_SHIFT CONFIG_SERIAL_AVR_RX_FIFO_SIZE
#include "rbtmpl.h"

static RB_T rx_rb;

static inline void enable_rx_irq(void) {

	out(UCSRB, in(UCSRB) | UCSRB_RXCIE);
}

static inline void disable_rx_irq(void) {

	out(UCSRB, in(UCSRB) & ~UCSRB_RXCIE);
}

static inline s16 noirq_getc(void) {
	s16 ch = -1;
	if (!rx_buffer_empty())
		ch = rx();
	return ch;
}

#if defined(CONFIG_WAKE_SEMA_FROM_IRQ) && defined(BLOCKING_RX)

struct completion rx_completion;

s16 avr232_getc(void) {
#ifndef RX_NOIRQCHECK
	if (irqs_disabled()) {
		return noirq_getc();
	}
#endif
	wait_for_completion(&rx_completion);
	/* FIXME: synchronize RB_GET_CHAR with a mutex here! */
	return RB_GET_CHAR(rx_rb);
}

static void avr232_rxirq(u8 irq) {
	u8 ch;

	ch = rx();

	if (rx_error())
		return;

// 	if (RB_FULL(rx_rb, ch)) {
// 		return;
// 	}
	if (RB_PUT_CHAR(rx_rb, ch) == 0) {
		complete_from_irq(&rx_completion);
	}
}

REQUEST_IRQ(IRQ_USART_RX, avr232_rxirq);


#else /* defined(CONFIG_WAKE_SEMA_FROM_IRQ) && defined(BLOCKING_RX) */

s16 avr232_getc(void) {
	irqflags_t flags;
	s16 ch;

	local_irq_save(flags);
	ch = RB_GET_CHAR(rx_rb);
	local_irq_restore(flags);
	return ch;
}

static void avr232_rxirq(u8 irq) {
	u8 ch;

	ch = rx();

	if (rx_error())
		return;
	RB_PUT_CHAR(rx_rb, ch);
}

REQUEST_IRQ_NO_RESCHED(IRQ_USART_RX, avr232_rxirq);

#endif /* defined(CONFIG_WAKE_SEMA_FROM_IRQ) && defined(BLOCKING_RX) */



static void avr232_rx_init(void) {
}

#else /* CONFIG_SERIAL_AVR_RX_IRQMODE */

s16 avr232_getc(void) {
	return getc_irqsave();
}

static void avr232_rx_init(void) {
}

#endif /* CONFIG_SERIAL_AVR_RX_IRQMODE */

#endif /* CONFIG_SERIAL_AVR_RX */



#ifdef CONFIG_SERIAL_AVR_CONSOLE

static void avr232_cons_putc(struct ocdevbase *dev, char ch) {

	if (ch == '\n')
		avr232_putc('\r');
	avr232_putc(ch);
}
struct ocdev kernel_console = {
	.out.putc_func = avr232_cons_putc,
};

#endif /* CONFIG_SERIAL_AVR_CONSOLE */

static int avr232_init(void) {
	u16 ucsrb = 0;

	// Set baud rate
	out(UBRRH, UBRR_VALUE >> 8);
	out(UBRRL, UBRR_VALUE & 0xff);

	out(UCSRA, 0);

#ifdef CONFIG_SERIAL_AVR_TX
	ucsrb |= UCSRB_TXEN;
#endif
#ifdef CONFIG_SERIAL_AVR_RX
	ucsrb |= UCSRB_RXEN;
#endif
#ifdef CONFIG_SERIAL_AVR_RX_IRQMODE
	ucsrb |= UCSRB_RXCIE;
#endif

	out(UCSRB, ucsrb);

#ifdef CONFIG_AVR_UART_NEW
	// Set frame format: 8data, 2stop bit
	out(UCSRC, UCSRC_UCSZ_MASK);
#else
	out(UCSRC, UCSRC_URSEL | UCSRC_USBS | UCSRC_UCSZ_MASK); // FIXME: Is this right?
#endif
	
// 	dbg_puts("avr232_init\n");
#ifdef CONFIG_SERIAL_AVR_TX
	avr232_tx_init();
#endif
#ifdef CONFIG_SERIAL_AVR_RX
	avr232_rx_init();
#endif
	return 0;
}

early_initcall(avr232_init);

