// SPDX-License-Identifier: GPL-2.0
// Copyright (c) 2010, 2017-2019 Rupert Eibauer. All rights reserved.

#include <linux/types.h>
#include <linux/interrupt.h>
#include <asm/io.h>


/* The AVR ADC can be used in 3 different modes:
- Single conversion allows multiple channels but active waiting is required.
  wait time can be shortened by disabling the power save option.
- interrupt mode allows multiple channels and passive waiting for results.
  wait time can be shortened by disabling the power save option.
- Free-Running mode allows only one channel but the result of the last value
  is always available.
*/
/* No interrupts yet */

#define ADC_READ_COUNT 1
#define MINIMUM_PRESCALER (CONFIG_CPUFREQ / 200000)
#define MAXIMUM_PRESCALER (CONFIG_CPUFREQ /  50000)

//#define ADPS_BITS (1 << ADPS1) | (1 << ADPS0)

/* The following cascade calculates the prescaler for fastest conversion. Not sure if a
   slower conversion is more accurate. */
#if MINIMUM_PRESCALER < 2
#error Clock too slow for ADC
#elif MINIMUM_PRESCALER < 4
#define ADPS_BITS ADCSRA_ADPS_2
#elif MINIMUM_PRESCALER < 8
#define ADPS_BITS ADCSRA_ADPS_4
#elif MINIMUM_PRESCALER < 16
#define ADPS_BITS ADCSRA_ADPS_8
#elif MINIMUM_PRESCALER < 32
#define ADPS_BITS ADCSRA_ADPS_16
#elif MINIMUM_PRESCALER < 64
#define ADPS_BITS ADCSRA_ADPS_32
#elif MINIMUM_PRESCALER < 128
#define ADPS_BITS ADCSRA_ADPS_64
#elif MAXIMUM_PRESCALER < 128
#define ADPS_BITS ADCSRA_ADPS_128
#else
#error Clock too fast for ADC
#endif

#ifdef CONFIG_ADC_SINGLE
static u16 __adc_read(void) {
	out(ADCSRA, in(ADCSRA) | ADCSRA_ADSC);
	while(in(ADCSRA) & ADCSRA_ADSC) {}
	return in_16(ADC);
}
#endif

static void __adc_set_channel(u8 channel, u8 refs) {
	out(ADMUX, (channel & ADMUX_MUX_MASK) | refs);
}
static void __adc_enable(void) {
	out(ADCSRA, ADCSRA_ADEN | ADPS_BITS);
}
#ifdef CONFIG_ADC_POWERSAVE
static void __adc_disable(void) {
	out(ADCSRA, in(ADCSRA) & ~ADCSRA_ADEN);
}
#endif

#ifdef CONFIG_ADC_SINGLE
u16 adc_read(unsigned char channel)
{
	int i=0;
	u16 result = 0;

	// spin_lock(&adc_lock);
	__adc_set_channel(channel, ADMUX_REFS_2V56);
#ifdef CONFIG_ADC_POWERSAVE
	__adc_enable();

	/* Warm-up ADC */
	__adc_read();
#endif
	for(i = 0; i < ADC_READ_COUNT; i++)
	{
		result += __adc_read();
	}

#ifdef CONFIG_ADC_POWERSAVE
	__adc_disable();
#endif
	// spin_unlock(&adc_lock);

	result /= ADC_READ_COUNT;
	return result;
}
#elif defined(CONFIG_ADC_FREE)

/* Lowest overhead, only one channel */
void adc_set_channel(u8 channel, u8 refs) {
	__adc_set_channel(channel, refs);
}
u16 adc_read(unsigned char channel)
{
	return in_16(ADC);
}

#elif defined(CONFIG_ADC_INT)

#include <linux/interrupt.h>
#include <linux/sched.h>

static u16   adc_result;
static struct completion adc_completion;
static struct mutex      adc_mutex;

static void adc_irq(u8 irq) {
    adc_result = in_16(ADC);
    out(ADCSRA, in(ADCSRA) & ~ADCSRA_ADIE);
    complete_from_irq(&adc_completion);
}

REQUEST_IRQ(ADC_vect_num, adc_irq);

u16 adc_read(unsigned char channel)
{
	u16 result;

	mutex_lock(&adc_mutex); /* Wait until the ADC resource is free */
	__adc_set_channel(channel, ADMUX_REFS_2V56);

	out(ADCSRA, in(ADCSRA) | ADCSRA_ADSC);
	wait_for_completion(&adc_completion);
	result = adc_result;
	mutex_unlock(&adc_mutex);

	return result;
}

static void __adc_init(void) {
    mutex_init(&adc_mutex);
    init_completion(&adc_completion);
}

#endif


void adc_init(void) {
#ifndef CONFIG_ADC_POWERSAVE
	__adc_enable();
#endif
#ifdef CONFIG_ADC_INT
	__adc_init();
#endif
#ifdef CONFIG_ADC_FREE
	out(ADCSRA, in(ADCSRA) | ADCSRA_ADFR);
#elif defined(CONFIG_ADC_INT)
	out(ADCSRA, in(ADCSRA) | ADCSRA_ADIE);
#endif
}
