<?php
// SPDX-License-Identifier: GPL-2.0
// Copyright (c) 2017-2019 Rupert Eibauer. All rights reserved.

include("FuseKconfig.inc.php");

function hweight8($x) {
	$x = ($x & 0x55) + (($x >> 1) & 0x55);
	$x = ($x & 0x33) + (($x >> 2) & 0x33);
	$x = ($x & 0x0f) + (($x >> 4) & 0x0f);
	return $x;
}

function mask_to_shift($msk) {
	$shift = 0;
	if ($msk == 0) {
		error_exit("Mask=0");
	}
	if (($msk & 0x0f) == 0) { $shift += 4; $msk >>= 4; }
	if (($msk & 0x03) == 0) { $shift += 2; $msk >>= 2; }
	if (($msk & 0x01) == 0) { $shift += 1; $msk >>= 1; }
	return $shift;
}

function error_exit($text) {
	echo "Error: $text\n";
	exit(1);
}

class FuseCfg {
	public $text, $name, $offset;
	public $default, $mask;
	public $shift = 0;
	function __construct($text, $name, $offset, $mask, $default) {
// 		echo "FuseCfg::construct name=$name offset=$offset mask=$mask default=$default\n";
		$this->text = $text; $this->name = $name; $this->offset = $offset;
		$this->default = $default & $mask;
		$this->mask = $mask;
		for ($i = 0; $i < 8; $i++) {
			if ($mask & (1 << $i))
				break;
		}
		$this->shift = $i;
	}
}
class BoolFuseCfg extends FuseCfg {
	function emitKconfig($Kconfig) {
		global $DEVICE;
		$Kconfig->addBool($this->name, $this->text, $DEVICE, $this->default ? "y" : "n");
	}
	function emitHeaderFile(&$fuse_byte, &$used_fuse_bits) {
		global $DEVICE;
		$ret = "";
		$ret .= "#ifdef CONFIG_".$this->name."\n";
		$ret .= "#define FUSE_".$this->name."_VALUE 0\n";
		$ret .= "#else\n";
		$ret .= "#define FUSE_".$this->name."_VALUE $this->mask\n";
		$ret .= "#endif\n";
		$ret .= "\n";
		$used_fuse_bits |= $this->mask;
		$fuse_byte["FUSE_".$this->name."_VALUE"] = true;
		return $ret;
	}
}

class EnumFuseCfg extends FuseCfg {
	public $options = [];
	private $optionNames = [];
	function addValue($text, $name, $value) {
// 		echo "enum $name value=$value shift=$this->shift\n";
		$i = 1;
		$suffix = "";
		while (isset($this->optionNames[$name.$suffix])) {
			$i++;
			$suffix = "_".$i;
		}
		if ($suffix != "") {
			echo "Warning: Eliminating duplicate $name with suffix $suffix\n";
		}
		$name .= $suffix;
		$this->optionNames[$name] = true;
		$this->options[$name] = [$text, $name, $value << $this->shift];
	}
	function emitHeaderFile(&$fuse_byte, &$used_fuse_bits) {
		global $DEVICE;
		$ret = "";
		foreach ($this->options AS $opt) {
			if ($ret == "")
				$ret .= "#ifdef CONFIG_".$this->name.'_'.$opt[1]."\n";
			else
				$ret .= "#elif defined(CONFIG_".$this->name.'_'.$opt[1].")\n";

			$ret .= "#define FUSE_".$this->name."_VALUE ".$opt[2]."\n";
			$used_fuse_bits |= $this->mask;
		}
		$ret .= "#else\n";
		$ret .= "#error\n";
		$ret .= "#endif\n";
		$ret .= "\n";
		$fuse_byte["FUSE_".$this->name."_VALUE"] = true;
		return $ret;
	}
	function emitKconfig($Kconfig) {
		global $DEVICE;
		$choice = $Kconfig->addChoice($this->name, $this->text, $DEVICE);
		foreach ($this->options AS $opt) {
			$name = $this->name."_".$opt[1];
			$choice->addBool($name, $opt[0], $DEVICE);
			if ($this->default == $opt[2])
				$choice->setDefault($name, $DEVICE);
			
		}
	}
}
class SplittableEnumOption {
	public $text, $name, $value, $depends = [];
	public $names = [];
	public $isDefault = false;
	static function compare($a, $b) {
		return $a->value - $b->value;
// 		return strcmp($a->name, $b->name);
	}
	function __construct($text, $name, $value, $depends = null) {
		$this->text = $text;
		$this->name = $name;
		$this->value = $value;
		if ($depends !== null)
			$this->depends[] = $depends;
	}
	function splitNames() {
		$name = explode('_', $this->name);
		$name_type = $name[0];
		$name_startup = "";
		switch($name_type) {
		case 'INTRCOSC':
		case 'WDOSC':
			for ($i = 1; $i < count($name); $i++) {
				if (substr($name[$i], -2) == "HZ")
					$name_type .= "_".$name[$i];
				else if ($name[$i] == "DEFAULT")
					continue;
				else if (substr($name[$i], -2) == "CK" || substr($name[$i], -2) == "MS" || substr($name[$i], -3) == "MS1") {
					if ($name_startup == "")
						$name_startup = $name[$i];
					else
						$name_startup .= "_".$name[$i];
				} else {
					error_exit("$name_type! ".$this->name);
				}
				
			}
			break;
		case 'EXTRCOSC':
			if (count($name) != 5) {
				error_exit("$name_type! ".$this->name);
			}
			$name_type .= '_'.$name[1].'_'.$name[2];
			$name_startup = $name[3].'_'.$name[4];
			break;
		case 'PLLCLK':
			if (count($name) == 5) {
				$name_type .= '_'.$name[1];
				$name_startup = $name[2].'_'.$name[3].'_'.$name[4];
			} else if (count($name) == 6) {
				$name_type .= '_'.$name[1].'_'.$name[2];
				$name_startup = $name[3].'_'.$name[4].'_'.$name[5];
			} else if (count($name) == 7) {
				$name_type .= '_'.$name[1].'_'.$name[2].'_'.$name[3];
				$name_startup = $name[4].'_'.$name[5].'_'.$name[6];
			} else if (count($name) == 8) {
				$name_type .= '_'.$name[1].'_'.$name[2].'_'.$name[3].'_'.$name[4];
				$name_startup = $name[5].'_'.$name[6].'_'.$name[7];
			} else {
				error_exit("$name_type! ".$this->name);
			}
			break;
		case 'RC':
			if ($name[2] == 'PLLIN' && $name[3] == 'RC' && count($name) == 8) {
				$name_type .= '_'.$name[1].'_'.$name[2].'_'.$name[3].'_'.$name[4];
				$name_startup = $name[5].'_'.$name[6].'_'.$name[7];
				break;
			}
			if (count($name) == 5) {
				$name_type .= '_'.$name[1];
				$name_startup = $name[2].'_'.$name[3].'_'.$name[4];
				break;
			}
			error_exit("$name_type! ".$this->name);
		case 'FSOSC':
			if (count($name) == 5) {
				$name_type .= '_'.$name[3].'_'.$name[4];
				$name_startup = $name[1].'_'.$name[2];
				break;
			}
			error_exit("$name_type! ".$this->name);
		case 'XOSC':
			if (count($name) == 7) {
				$name_type .= '_'.$name[1].'_'.$name[2].'_'.$name[3];
				$name_startup = $name[4].'_'.$name[5].'_'.$name[6];
				break;
			}
			if (count($name) == 8) {
				$name_type .= '_'.$name[1].'_'.$name[2].'_'.$name[3].'_'.$name[4];
				$name_startup = $name[5].'_'.$name[6].'_'.$name[7];
				break;
			}
			if (count($name) == 9) {
				$name_type .= '_'.$name[1].'_'.$name[2].'_'.$name[3].'_'.$name[4].'_'.$name[5];
				$name_startup = $name[6].'_'.$name[7].'_'.$name[8];
				break;
			}
			error_exit("$name_type! ".$this->name);
		case 'EXTLOFXTALRES':
		case 'EXTMEDFXTALRES':
		case 'EXTHIFXTALRES':
		case 'EXTLOFXTAL':
		case 'EXTCLK':
		case 'TOSC':
		case 'TRXOSC':
			if ($name_type == "EXTLOFXTAL" && $name[count($name)-1] == "INTCAP") {
				$name_type .= "_INTCAP";
				unset($name[count($name)-1]);
			}
			if ($name_type == "EXTCLK" && count($name) == 7) {
				$name_type = $name[1].'_'.$name[2].'_'.$name[3];
				$name_startup = $name[4].'_'.$name[5].'_'.$name[6];
				break;
			}
			if (count($name) != 3 && count($name) != 4) {
				error_exit("$name_type! ".$this->name);
			}
			$name_startup = $name[1].'_'.$name[2];
			if (count($name) == 4)
				$name_startup .= "_".$name[3];
			break;
		case 'EXTFSXTAL':
			if (count($name) != 4) {
				error_exit("$name_type! ".$this->name);
			}
			$name_startup = $name[1].'_'.$name[2].'_'.$name[3];
			break;
		case 'EXTXOSC':
			if (count($name) != 6 && count($name) != 5) {
				error_exit("$name_type! ".$this->name);
			}
			$name_type .= '_'.$name[1].'_'.$name[2];
			$name_startup = $name[3].'_'.$name[4];
			if (count($name) == 6)
				$name_startup .= '_'.$name[5];
			break;
		case '14CK':
			$name_type = "";
			$name_startup = $name[0].'_'.$name[1];
			break;
		default:
			echo $this->text."\n";
			print_r($name);
			error_exit("$name_type! ".$this->name);
		}
		$this->names[0] = $name_type;
		$this->names[1] = $name_startup;
	}
	function setDefault() {
		$this->isDefault = true;
	}
	function addDepend($depend) {
		$this->depends[] = $depend;
	}
}

class EnumFuseClkCfg extends FuseCfg {
	public $options = [];
	function addValue($text, $name, $value, $depends = null) {
		if (!isset($this->options[$name])) {
// 			$this->options[$name] = new SplittableEnumOption($text, $name, ($value << $this->shift) ^ $this->mask);
			$this->options[$name] = new SplittableEnumOption($text, $name, $value << $this->shift);
		}// else check?
		if ($depends !== null) {
			$this->options[$name]->addDepend($depends);
		}
	}
	function setDefault($name) {
		$this->options[$name]->setDefault();
	}
	public $subchoices;
	public $dependencies;
	function createSubChoices() {
		$this->subchoices['CLOCK'] = new EnumFuseClkCfg('Clock Type', 'CKSEL', $this->offset, 0, 0);
		$this->subchoices['STARTUP'] = new EnumFuseClkCfg('Clock Start-up time', 'SUT', $this->offset, 0, 0);
		foreach ($this->options AS $opt) {
			$parts = explode('; ', $opt->text);
			$default = false;
			if ($parts[count($parts)-1] == 'default value') {
				$default = true;
// 				$opt->setDefault();
				unset($parts[count($parts)-1]);
			}
			$opt->splitNames();
			$name_type = $opt->names[0];
			$name_startup = $opt->names[1];
			
			global $Device;
			if ($Device == "AT90PWM161" || $Device == "AT90PWM81.atdf") {
				$parts = explode("SUT:", $opt->text);
			}
			$desc_type = $parts[0];
			
			if ($name_type == "") {
				$this->subchoices['STARTUP']->addValue($opt->text, $name_startup, 0);
				if ($default) {
					$this->subchoices['STARTUP']->setDefault($name_startup);
				}
			} else {
				$desc_startup = explode(': ', $parts[1])[1];
				if ($Device == "AT90PWM161" || $Device == "AT90PWM81.atdf") {
					$desc_startup = trim(explode('; ', $parts[1])[0]);
// 					$desc_startup = $parts[1];
				}
				$this->subchoices['CLOCK']->addValue($desc_type, $name_type, 0);
				$this->subchoices['STARTUP']->addValue($desc_startup, $name_startup, 0, 'CKSEL_'.$name_type);
				if ($default) {
					$this->subchoices['CLOCK']->setDefault($name_type);
					$this->subchoices['STARTUP']->setDefault($name_startup);
				}
			}
		}
		/* Finally, sort ourselves. */
		usort($this->options, array('SplittableEnumOption', 'compare'));
	}
	function emitHeaderFile(&$fuse_byte, &$used_fuse_bits) {
		global $DEVICE;
		$ret = "";
		$def = "";
		foreach ($this->options AS $opt) {
			$cond = "defined(CONFIG_CKSEL_".$opt->names[0].") && defined(CONFIG_SUT_".$opt->names[1].")";
			if ($ret == "")
				$ret .= "#if $cond\n";
			else
				$ret .= "#elif $cond\n";

			$ret .= "#define FUSE_".$this->name."_VALUE $opt->value\n";
		}
		$fuse_byte["FUSE_".$this->name."_VALUE"] = true;
		$ret .= "#else\n";
		$ret .= "#error\n";
		$ret .= "#endif\n";
		$ret .= "\n";
		$used_fuse_bits |= $this->mask;
		return $ret;
	}
	function emitKconfig($Kconfig) {
		global $DEVICE;
// 		echo "EnumFuseClkCfg::emitKconfig() mask=$this->mask\n";
		if ($this->mask != 0) {
			$ret = $this->subchoices['CLOCK']->emitKconfig($Kconfig);
			$ret .= $this->subchoices['STARTUP']->emitKconfig($Kconfig);
			return $ret;
		}
		
		$choice = $Kconfig->addChoice($this->name, $this->text, $DEVICE);
		foreach ($this->options AS $opt) {
			$name = $this->name."_".$opt->name;
			$bool = $choice->addBool($name, $opt->text, $DEVICE, null, implode(" || ", $opt->depends));
			if ($opt->isDefault)
				$choice->setDefault($name, $DEVICE);
			$optnames = explode("_", $opt->name);
			if ($this->name == "CKSEL" && $optnames[0] == "INTRCOSC" && count($optnames) == 2) {
				global $fusecfg_cpufreq;
				$hz = 0;
				if (substr($optnames[1], -3) == "MHZ") {
					$hz = (int)($optnames[1]) * 1000000;
				} else if (substr($optnames[1], -3) == "KHZ") {
					$hz = (int)($optnames[1]) * 1000;
				} 
				if ($hz != 0) {
					$bool->addSelect("CPUFREQ_IS_FIXED");
					$bool->selectCpufreq($hz);
				}
			}
		}
	}
}

function parseRegisterGroup($node) {
	$configs = [];
	foreach ($node->childNodes AS $child) {
		if ($child->nodeName == "#text")
			continue;
		$reg_offset = decode_number($child->getAttribute('offset'));
		$reg_name = $child->getAttribute('name');
		$reg_size = decode_number($child->getAttribute('size')); // Should be 1
		$reg_initval = decode_number($child->getAttribute('initval')) ^ 0xff;
// 		echo "CHILD2: <".$child->nodeName.$attrs.">\n";
		foreach ($child->childNodes AS $bitfield) {
			if ($bitfield->nodeName == "#text")
				continue;
			$field_caption = $bitfield->getAttribute('caption');
			$field_name = $bitfield->getAttribute('name');
			$field_mask = decode_number($bitfield->getAttribute('mask'));
// 			echo "BITFIELD: name=$field_name offset=$reg_offset mask=$field_mask init=$reg_initval\n";
			if ($bitfield->hasAttribute('values')) {
				$field_values = $bitfield->getAttribute('values');
// 				echo $field_name."\n";
				if ($field_name == 'CKSEL_SUT') {
					$field_name = 'SUT_CKSEL';
				}
				if ($field_name == 'SUT_CKSEL') {
// 					echo "EnumFuseClkCfg\n";
					$configs[$field_values] = new EnumFuseClkCfg($field_caption, $field_name, $reg_offset, $field_mask, $reg_initval);
				} else {
					$configs[$field_values] = new EnumFuseCfg($field_caption, $field_name, $reg_offset, $field_mask, $reg_initval);
				}
			} else
				$configs[".".$field_name] = new BoolFuseCfg($field_caption, $field_name, $reg_offset, $field_mask, $reg_initval);
		}
	}
	return $configs;
}

/*********************** FUSES Kconfig ***********************/
function decode_fuses($fusesModule) {
	global $groups;

	foreach ($fusesModule->childNodes AS $child) {
		if ($child->nodeName == "#text")
			continue;
		if ($child->nodeName == "register-group") {
			$groups = parseRegisterGroup($child);
		} else if ($child->nodeName == "value-group") {
			$group = $groups[$child->getAttribute('name')];
			foreach ($child->childNodes AS $valueNode) {
				if ($valueNode->nodeName == "#text")
					continue;
				$name = $valueNode->getAttribute('name');
				$caption = $valueNode->getAttribute('caption');
				if ($group->name == "BOOTSZ") {
					$name = explode("_", $name);
					$name = $name[0];
					$caption = substr($name, 0, -1)." words";
				}
				$group->addValue($caption, $name, decode_number($valueNode->getAttribute('value')));
			}
		}
	}

	global $Device, $device, $DEVICE, $fusecfg_cpufreq;
	if (file_exists("Kconfig.ser")) {
		$Kconfig = unserialize(file_get_contents("Kconfig.ser"));
	} else {
		$Kconfig = new FuseKconfig();
	}
	$Kconfig->addDevice($DEVICE);
	$groups = array_reverse($groups);
	foreach ($groups AS $group) {
		if (get_class($group) == 'EnumFuseClkCfg') {
			$group->createSubChoices();
		}
		$group->emitKconfig($Kconfig);
		
	}

	file_put_contents("Kconfig.ser", serialize($Kconfig));
}


if (basename($argv[0]) == "fuseconfig.php") {
	$Kconfig = unserialize(file_get_contents("Kconfig.ser"));

	$fuseKconfig = "";
	$fuseKconfig .= "\n";
	$fuseKconfig = $Kconfig->emit();
	file_put_contents("devs/fuses.Kconfig", $fuseKconfig);

	$freqsel = $Kconfig->emit_cpufreq();
	file_put_contents("devs/cpufreq.Kconfig", $freqsel);
}
