<?php
// SPDX-License-Identifier: GPL-2.0
// Copyright (c) 2017-2019 Rupert Eibauer. All rights reserved.

function decode_number($num) {
	if (substr($num, 0, 2) == "0x")
		return hexdec(substr($num, 2));
	return $num + 0;
}

require_once("fuseconfig.php");

/* Main program */
$Device = $argv[1];
$DEVICE = strtoupper($Device);
$device = strtolower($Device);

$file = file_get_contents("atdf/$Device.atdf");
$file = preg_replace('/<avr-tools-device-file [^>]*>/', '<avr-tools-device-file xmlns="urn:avr-tools-device-file">', $file);
$doc = new DOMDocument();
$doc->loadXML($file);

/*********************** Pre-parse XML ***********************/

$modules = $doc->getElementsByTagName("modules")->item(0)->childNodes;
$otherModules = array();
$valueGroups = array();
foreach ($modules AS $module) {
	if ($module->nodeName == "#text")
		continue;

	if ($module->getAttribute("name") == "FUSE") {
		$fusesModule   = $module;
	} else {
		$otherModules[] = $module;
	}
	$vgs = $module->getElementsByTagName("value-group");
	foreach ($vgs AS $vg) {
		$valueGroups[$vg->getAttribute("name")] = $vg;
	}
}

/*********************** FUSES Kconfig ***********************/

decode_fuses($fusesModule);

/*********************** irqvec.h ***********************/

$interrupsNode = $doc->getElementsByTagName("interrupts")->item(0);
$irqvec_h  = "";
$irqvec_h .= "#ifndef _AVR_IRQVEC_".$DEVICE."_H_\n";
$irqvec_h .= "#define _AVR_IRQVEC_".$DEVICE."_H_\n";
$irqvec_h .= "\n";
$irqvec_h .= "/* DO NOT EDIT. generated by decode_pack.php */\n";
$irqvec_h .= "\n";
$irqvec_h1 = "";
$irqvec_h2 = "";
$max_index = 0;
foreach ($interrupsNode->childNodes AS $child) {
	if ($child->nodeName != "interrupt")
		continue;
	$index = $child->getAttribute("index");
	$name = $child->getAttribute("name");
	$irqvec_h1 .= sprintf("#define IRQ_%-12s  %3d\n", $name, $index);
	$irqvec_h2 .= sprintf("#define IRQ_NAME_%-3d %s\n", $index, $name);
	if ($max_index < $index)
		$max_index = $index;
}
$irqvec_h .= $irqvec_h1;
$irqvec_h .= "\n";
$irqvec_h .= $irqvec_h2;
$irqvec_h .= "\n";
$irqvec_h .= "#define NR_IRQS $max_index\n"; // Reset vector does not count here.
$irqvec_h .= "\n";
$irqvec_h .= "#endif /* _AVR_IRQVEC_".$DEVICE."_H_ */\n";

$vectors_h = "";
for ($i = 1; $i <= $max_index; $i++) {
	$vectors_h .= "VECTOR($i)\n";
}


/*********************** ioregs.h ***********************/

$prefix = "";
$ioregs_h  = "";
$ioregs_h .= "#ifndef _AVR_IOREGS_".$DEVICE."_H_\n";
$ioregs_h .= "#define _AVR_IOREGS_".$DEVICE."_H_\n";
$ioregs_h .= "\n";
$ioregs_h .= "/* DO NOT EDIT. generated by decode_pack.php */\n";
$ioregs_h .= "\n";
$strfmt = "%-30s";
foreach ($otherModules AS $mod) {
	$reg_groups = $mod->getElementsByTagName("register-group");
	foreach ($reg_groups AS $reg_group) {
		$ioregs_h .= "\n/* Register group ".$reg_group->getAttribute("caption")." */\n\n";
		$regs = $reg_group->getElementsByTagName("register");
		foreach ($regs AS $reg) {
			$regname = $reg->getAttribute("name");
			$offset = decode_number($reg->getAttribute("offset"));
			$regsize = $reg->getAttribute("size");
			$caption = $reg->getAttribute("caption");
			if ($caption != "") $caption = "  /* $caption */";

			$ioregs_h .= sprintf("#define $prefix$strfmt 0x%02x%s\n", $regname, $offset, $caption);
			if ($regsize > 1) {
				$suffix = array("L", "H", "HL", "HH");
				for ($i = 0; $i < $regsize; $i++) {
					$ioregs_h .= sprintf("#define $prefix$strfmt 0x%02x\n", $regname.$suffix[$i], $offset + $i);
				}
			}

			$bitfields = $reg->getElementsByTagName("bitfield");
			foreach ($bitfields AS $bitfield) {
				$name = $bitfield->getAttribute("name");
				$mask = decode_number($bitfield->getAttribute("mask"));
				if ($regname == "TCCR0") {
					if ($name == "CS02" || $name == "CS01") {
// 						echo "CS01 and CS00 detected\n";
						continue;
					}
					if ($name == "CS00") {
						$mask |= ($mask << 1) | ($mask << 2); // FIXME: Check CS02 and CS01
						$name = "CS0";
					}
				}
				$macros_used = array();
				$caption = $bitfield->getAttribute("caption");
				if (hweight8($mask) != 1) {
					if ($bitfield->hasAttribute("values")) {
						$ioregs_h .= "\n";
						$ioregs_h .= sprintf("#define $prefix$strfmt 0x%02x  /* %s */\n", $regname."_".$name."_MASK", $mask, $caption);
						$shift = mask_to_shift($mask);
						$values = $bitfield->getAttribute("values");

						$values = $valueGroups[$values]->getElementsByTagName("value");

						foreach ($values AS $value) {
							$valname = $value->getAttribute("name");
							$valval = decode_number($value->getAttribute("value"));
							$valval <<= $shift;
							$caption = $value->getAttribute("caption");
							if ($name == "CS1" || $name == "CS0" || $name == "CS2") {
								switch ($caption) {
									case "No clock source (Timer/Counter stopped)":
									case "No clock source (Timer/Counter2 stopped)":
									case "No clock source (Timer/Counter0 stopped)":
									case "No Clock Source (Stopped)": $valname = "STOPPED"; break;
									case "clk_IO/1 (no prescaling)":
									case "clk_T2S/1 (no prescaling)":
									case "clk_IO/1 (no prescaling)":
									case "Running, No Prescaling": $valname = "PS_1"; break;
									case "External clock source on Tn pin, clock on falling edge":
									case "External clock source on T0 pin, clock on falling edge":
									case "Running, ExtClk Tx Falling Edge": $valname = "EXTCLK_FALLING"; break;
									case "External clock source on Tn pin, clock on rising edge":
									case "External clock source on T0 pin, clock on rising edge":
									case "Running, ExtClk Tx Rising Edge": $valname = "EXTCLK_RISING"; break;
									default:
										if (substr($caption, 0, 13) == "Running, CLK/") {
											$valname = "PS_".substr($caption, 13);
											break;
										} else if (substr($caption, 0, 7) == "clk_IO/" && substr($caption, -17) == " (from prescaler)") {
											$valname = "PS_".substr($caption, 7, -17);
											break;
										} else if (substr($caption, 0, 8) == "clk_T2S/" && substr($caption, -17) == " (from prescaler)") {
											$valname = "PS_".substr($caption, 8, -17);
											break;
										}
										error_exit("Unknown $name value $caption");
								}
							} else if ($name == "ISC1" || $name == "ISC0") {
								switch ($caption) {
									case "The low level of INTn generates an interrupt request.":
									case "Any edge of INTn generates asynchronously an interrupt request.":
									case "The falling edge of INTn generates asynchronously an interrupt request.":
									case "The rising edge of INTn generates asynchronously an interrupt request.":
									
						
									case "Low Level of INTX": $valname = "LOW"; break;
									case "Any Logical Change of INTX": $valname = "ANY"; break;
									case "Any Logical Change in INTX": $valname = "ANY"; break;
									case "Falling Edge of INTX": $valname = "FALLING"; break;
									case "Rising Edge of INTX": $valname = "RISING"; break;
									case "Reserved": $valname = ""; break;
									default: error_exit("Unknown $name value \"$caption\"");
								}
							} else if ($name == "REFS") {
								switch ($caption) {
									case "AREF, Internal reference voltage generation turned off":
									case "AREF, Internal Vref turned off": $valname = "AREF"; break;
									case "AVDD with external capacitor at AREF pin": $valname = "AVDD"; break;
									case "AVCC with external capacitor at AREF pin": $valname = "AVCC"; break;
									case "Reserved": $valname = ""; break;
									case "Internal 1.1V Voltage Reference with external capacitor at AREF pin": $valname = "1V1"; break;
									case "Internal 1.5V Voltage Reference (no external capacitor at AREF pin)": $valname = "1V5"; break;
									case "Internal 1.6V Voltage Reference (no external capacitor at AREF pin)": $valname = "1V6"; break;
									case "Internal 2.56V Voltage Reference with external capacitor at AREF pin": $valname = "2V56"; break;
									default: error_exit("Unknown $name value $caption");
								}
							} else if ($name == "ADPS" || $name == "TWPS") {
								if ((string)(int)$caption !== (string)$caption) {
									error_exit("Unknown $name value $caption");
								}
								$valname = $caption;
							} else if ($name == "WDP") {
								if (substr($caption, 0, 18) !== "Oscillator Cycles ") {
									error_exit("Unknown $name value $caption");
								}
								$valname = substr($caption, 18);
							} else if ($name == "UPM") {
								switch ($caption) {
									case "Disabled": $valname = "DISABLED"; break;
									case "Enabled, Even Parity": $valname = "EVEN"; break;
									case "Enabled, Odd Parity": $valname = "ODD"; break;
									case "Reserved": break;
									default: erro_exit("Unknown $name value $caption");
								}
							} else if ($name == "ACIS") {
								switch ($caption) {
									case "Interrupt on Toggle": $valname = "TOGGLE"; break;
									case "Interrupt on Falling Edge": $valname = "FALLING"; break;
									case "Interrupt on Rising Edge": $valname = "RISING"; break;
									case "Reserved": break;
									default: error_exit("Unknown $name value $caption");
								}
							} else if ($name == "SPR") {
								if (substr($caption, 0, 5) !== "fosc/") {
									error_exit("Unknown $name value $caption");
								}
								$valname = substr($caption, 5);
							}
							if ($valname != "") {
								$valname = $regname."_".$name."_".$valname;
								if (isset($macros_used[$valname]) && $macros_used[$valname] != $valval) {
									$dup = 0;
									while (isset($macros_used[$valname."_DUP$dup"]) && $macros_used[$valname] != $valval) {
										$dup++;
									}
									$valname .= "_DUP$dup";
								}
								if (!isset($macros_used[$valname])) {
									$macros_used[$valname] = $valval;
									$ioregs_h .= sprintf("#define $prefix$strfmt 0x%02x  /* %s */\n", $valname, $valval, $caption);
								}
							}
						}
					} else {
						$ioregs_h .= sprintf("#define $prefix$strfmt 0x%02x  /* %s */\n", $regname."_".$name."_MASK", $mask, $caption);
					}
				} else {
					$ioregs_h .= sprintf("#define $prefix$strfmt 0x%02x  /* %s */\n", $regname.'_'.$name, $mask, $caption);
					$ioregs_h .= sprintf("#define $prefix$strfmt 0x%02x\n", $regname.'_'.$name."_BIT", mask_to_shift($mask));
				}
			}
			$ioregs_h .= "\n";
		}
	}
}
$ioregs_h .= "#endif /* _AVR_IOREGS_".$DEVICE."_H_ */\n";

/*********************** limits.h ***********************/
$num_fuse_bytes = 0;

$memorySegmentNodes = $doc->getElementsByTagName("memory-segment");
$limits_h  = "";
$limits_h .= "#ifndef _AVR_LIMITS_".$DEVICE."_H_\n";
$limits_h .= "#define _AVR_LIMITS_".$DEVICE."_H_\n";
$limits_h .= "\n";
$limits_h .= "/* DO NOT EDIT. generated by decode_pack.php */\n";
$limits_h .= "\n";
foreach ($memorySegmentNodes AS $child) {
	if ($child->nodeName != "memory-segment")
		continue;
	$name = $child->getAttribute("name");
	if ($name != "IRAM" && $name != "FLASH" && $name != "EEPROM" && $name != "FUSES" && $name != "LOCKBITS")
		continue;
	if ($name == "IRAM")
		$name = "RAM";

	$start_name = $name."_START";
	$end_name = $name."_END";
	$start_value = decode_number($child->getAttribute("start"));
	$end_value = $start_value + decode_number($child->getAttribute("size")) - 1;
	if ($name == "FUSES")
		$num_fuse_bytes = $end_value + 1;

	$limits_h .= sprintf("#define %-15s 0x%x\n", $start_name, $start_value);
	$limits_h .= sprintf("#define %-15s 0x%x\n", $end_name, $end_value);
	$limits_h .= "\n";
	if ($name == "FLASH" || $name == "EEPROM") {
		$pagesize = decode_number($child->getAttribute("pagesize"));
		$limits_h .= sprintf("#define %-15s 0x%x\n", $name."_PAGESIZE", $pagesize);
		$limits_h .= "\n";
	}
}
$limits_h .= "#endif /* _AVR_LIMITS_".$DEVICE."_H_ */\n";

/*********************** fusecalc.h ***********************/

$fusecalc_h  = "#ifndef _AVR_FUSECALC_".$DEVICE."_H\n";
$fusecalc_h .= "#define _AVR_FUSECALC_".$DEVICE."_H\n";
$fusecalc_h .= "\n";
$fusecalc_h .= "/* DO NOT EDIT. generated by decode_pack.php */\n";
$fusecalc_h .= "\n";
$fuse_bytes = [];
$used_fuse_bits = [];
for ($i = 0; $i < $num_fuse_bytes; $i++) {
	$fuse_bytes[] = [];
	$used_fuse_bits[] = 0;
}
foreach ($groups AS $group) {
	if ($group->offset >= $num_fuse_bytes) {
		error_exit("ILLEGAL FUSE OFFSET $group->offset!");
	}
	$fusecalc_h .= $group->emitHeaderFile($fuse_bytes[$group->offset], $used_fuse_bits[$group->offset]);
}
for ($i = 0; $i < $num_fuse_bytes; $i++) {
	$mask = 0xff & ~$used_fuse_bits[$i];
	$value = sprintf("0x%02x", $mask);
	if (count($fuse_bytes[$i]))
		$value = "(".$value." | ".implode(' | ', array_keys($fuse_bytes[$i])).")";
	
	$fusecalc_h .= "#define FUSE_BYTE_$i $value\n";
}
$fusecalc_h .= "\n";
$fusecalc_h .= "#endif /* _AVR_FUSECALC_".$DEVICE."_H */\n";

/*********************** gpios.h ***********************/

$peripherals = $doc->getElementsByTagName("peripherals");

$peripheral_modules = $peripherals[0]->getElementsByTagName("module");
foreach ($peripheral_modules AS $module) {
	if ($module->getAttribute("name") == "PORT") {
		$portModule = $module;
		break;
	}
}
$gpios_h  = "";
$gpios_h .= "#ifndef _AVR_GPIOS_".$DEVICE."_H\n";
$gpios_h .= "#define _AVR_GPIOS_".$DEVICE."_H\n";
$gpios_h .= "\n";
$gpios_h .= "/* DO NOT EDIT. generated by decode_pack.php */\n";
$gpios_h .= "\n";
$gpios = [];
foreach ($portModule->getElementsByTagName("signal") AS $signal) {
	$pad = $signal->getAttribute("pad");
	$port = $pad[1];
	$bit = $pad[2];
	if (!isset($gpios[$port]))
		$gpios[$port] = [];
	$gpios[$port][$bit] = true;
}
foreach ($gpios AS $port => $bits) {
	foreach ($bits AS $bit => $true) {
		$gpios_h .= "#define GPIO_${port}${bit}_PORT   PORT${port}\n";
		$gpios_h .= "#define GPIO_${port}${bit}_DDR    DDR${port}\n";
		$gpios_h .= "#define GPIO_${port}${bit}_PIN    PIN${port}\n";
		$gpios_h .= "#define GPIO_${port}${bit}_BITNUM ${bit}\n";
		$gpios_h .= "#define GPIO_${port}${bit}_NBITS  ".count($bits)."\n";
		$gpios_h .= "\n";
	}
}
$gpios_h .= "#endif /* _AVR_GPIOS_".$DEVICE."_H */\n";

/*********************** Finally, write everything to the files ... ***********************/

file_put_contents("devs/allavr.Kconfig", "config $DEVICE\n\tbool \"$Device\"\n\n", FILE_APPEND);
// file_put_contents("devs/allfuses.Kconfig", "source \"arch/avr/devs/$device/Kconfig\"\n", FILE_APPEND);
file_put_contents("devs/devices.Makefile", "cpu-$(CONFIG_$DEVICE) := $device\n", FILE_APPEND);

mkdir("devs/$device/include/asm", 0755, true);
file_put_contents("devs/$device/include/asm/fusecalc.h", $fusecalc_h);
file_put_contents("devs/$device/include/asm/irqvec.h", $irqvec_h);
file_put_contents("devs/$device/include/asm/ioregs.h", $ioregs_h);
file_put_contents("devs/$device/include/asm/vectors.h", $vectors_h);
file_put_contents("devs/$device/include/asm/limits.h", $limits_h);
file_put_contents("devs/$device/include/asm/gpios.h", $gpios_h);

exit(0);
