<?php
// SPDX-License-Identifier: GPL-2.0
// Copyright (c) 2017-2019 Rupert Eibauer. All rights reserved.

function text_convert($r) {
	if (strpos($r, "[") === false && substr($r, -1) == "]")
		$r = substr($r, 0, -1);
	$replace = [
		"detection level at" => "detection at",
		"Watch-dog Timer"    => "Watchdog Timer",
		"Watchdog timer"     => "Watchdog Timer",
		"Brown-out"          => "Brownout",
		"detector"           => "Detector",
		"Detection"          => "detection",
		"Disabled"           => "disabled",
		"Select "            => "",
		"boot size"          => "Boot Size",
	];
	$r = str_replace(array_keys($replace), array_values($replace), $r);
	switch ($r) {
		case "Startup Time":
		case "start-up time": $r = "Clock Start-up time"; break;
	}
	return $r;
}

function std_text($text, $name) {
	switch ($name) {
		case "RSTDISBL": $text = "Reset disabled"; break;
		case "CKOUT": $text = "Clock output"; break;
	}
	return $text;
}

class KconfigBool {
	protected $root;
	protected $order;
	protected $after = [];
	function __construct($root, $name, $text) {
		$this->root = $root;
		$this->name = $name;
		$this->text = $text;
	}
	public $name, $text, $devices = [], $select = [];
	function addDevice($dev, $default, $deps = "", $text = "") {
		$this->devices[$dev] = [];
		if ($default !== null)
			$this->devices[$dev]["default"] = $default;
		if ($deps != "")
			$this->devices[$dev]["depends"] = $deps;
		if ($text != "" && $text != $this->text) {
// 			echo "Warning: different text \"$text\" for device $dev\n";
			$this->devices[$dev]["text"] = $text;
		}
	}
	function findChoice($name) {
		return null;
	}
	function addSelect($name, $dev = null) {
		if ($dev == null) {
			$this->select[$name] = true;
		} else {
			if (!isset($this->devices[$dev]["select"]))
				$this->devices[$dev]["select"] = [];
			$this->devices[$dev]["select"][$name] = true;
		}
	}
	function selectCpufreq($hz) {
		$this->root->selectCpufreq($this->name, $hz);
	}
	function collect_depends() {
		$deps = [];
		foreach ($this->devices AS $devname => $dev) {
			if (isset($dev["depends"])) {
				$depends = $dev["depends"];
			} else
				$depends = "";
			if (!isset($deps[$depends])) {
				$deps[$depends] = [$devname => true];
			} else {
				$deps[$depends][$devname] = true;
			}
		}
		return $deps;
	}
	function create_depends_list() {
		$deparr = $this->collect_depends();
		
		$depends = "";
		$last_linebreak = 0;
		foreach ($deparr AS $deps => $devices) {
			if (strlen($depends) - $last_linebreak > 40) {
				$depends .= " \\\n\t       ";
				$last_linebreak = strlen($depends);
			}
			if ($depends != "")
				$depends .= " || ";

			$neg_devices = false;
			/* In case we depend on a list of other options (or-combined) */
			if ($deps != "" && strpos($deps, "&&") === false) {
				$neg_devlist = [];
				// We scan all choices we depend on
				// and if they have the same devices list, or a more restrictive one
				// we we can ignore our own devices list.
				foreach (explode(" || ", $deps) AS $dep) { // FIXME: We assume it is cksel, and we are SUT for others we don't have dependencies yet.
					$choice = $this->root->findChoice($dep);
					foreach ($choice->devices AS $dev => $dummy) {
						if (!isset($devices[$dev])) {
							// or just create an alternative list
							$neg_devlist[$dev] = true;;
						}
					}
				}
				$neg_devices = "";
				if (count($neg_devlist))
					$neg_devices = "!".implode(" && !", array_keys($neg_devlist));
			}
			$devices = $this->root->dev_depends_list($devices);
			if ($neg_devices !== false && strlen($neg_devices) < strlen($devices))
				$devices = $neg_devices;

			if ($devices == "") {
				if (count($deparr) == 1)
					return $deps;
				else
					$depends .= "(".$deps.")";
			} else if ($deps == "") {
				$depends .= $devices;
			} else {
				if (strpos($devices, "||") !== false) {
					$devices = "($devices)";
				}
				if (strpos($deps, "||") !== false) {
					$deps = "($deps)";
				}
				$d = "$devices && $deps";
				if (count($deparr) > 1)
					$d = "($d)";
				$depends .= $d;
			}
			
		}
		return $depends;
	}
	function emitDependsAndDefaults() {
		$r = "";
		$depends = $this->create_depends_list();
		if ($depends != "")
			$r .= "\tdepends on $depends\n";
		$defaults = [];
		foreach ($this->devices AS $dev => $devcfg) {
			if (isset($devcfg["default"])) {
				if (!isset($defaults[$devcfg["default"]])) {
					$defaults[$devcfg["default"]] = [];
				}
				$defaults[$devcfg["default"]][$dev] = true;
			}
		}
		foreach($defaults AS $value => $devices) {
			if ($value == "n")
				continue;
			$devices = $this->root->dev_depends_list($devices);
			if ($devices == "" || $devices == $depends) {
				$r .= "\tdefault $value\n";
			} else {
				$r .= "\tdefault $value if $devices\n";
			}
		}
		return $r;
	}
	function emitSelects() {
		$r = "";
		if (count($this->select)) {
			foreach ($this->select AS $select => $true)
				$r .= "\tselect $select\n";
		}
		$selects = [];
		foreach ($this->devices AS $dev => $devcfg) {
			if (isset($devcfg["select"])) {
				foreach ($devcfg["select"] AS $select => $true) {
					if (!isset($selects[$select]))
						$selects[$select] = [$dev => true];
					else
						$selects[$select][$dev] = true;
				}
			}
		}
		foreach ($selects AS $select => $devs) {
			$devs = $this->root->dev_depends_list($devs);
			if ($devs != "")
				$r .= "\tselect $select if $devs\n";
			else
				$r .= "\tselect $select\n";
		}
		return $r;
	}
	function addHelp() {
		$prompts = [];
		$r = "";
		foreach ($this->devices AS $dev => $devcfg) {
			if (isset($devcfg["text"])) {
				$text = $devcfg["text"];
				$r .= "\t  Different description for $dev: \"$text\"\n";
			}
		}
		if ($r != "")
			$r = "\thelp\n".$r;
		return $r;
	}
	function emit() {
		$r = "";
		$r .= "config $this->name\n";
		$r .= "\tbool \"$this->text\"\n";

		$r .= $this->emitDependsAndDefaults();

		$r .= $this->emitSelects();
		$r .= $this->addHelp();
		$r .= "\n";
		return $r;
		
	}
}

class KconfigChoice extends KconfigBool {
	private $choices;
	private $devOrder = [];
	function findChoice($name) {
		if (isset($this->choices[$name]))
			return $this->choices[$name];
		return null;
	}
	function addBool($name, $text, $device, $default = null, $deps = "") {
		$text = text_convert($text);
		if (!isset($this->devOrder[$device]))
			$this->devOrder[$device] = [$name => true];
		else
			$this->devOrder[$device][$name] = true;

		if (!isset($this->choices[$name])) {
			$this->choices[$name] = new KconfigBool($this->root, $name, $text, std_text($text, $name));
		}
		$this->choices[$name]->addDevice($device, $default, $deps, $text);
		return $this->choices[$name];
	}
	function setDefault($default, $device) {
		$this->devices[$device]["default"] = $default;
	}


	static function cmp_order($a, $b) {
		return count($a->after) - count($b->after);
		if (isset($a->after[$b->name])) {
			if (isset($b->after[$a->name])) {
				echo "Warning: seems like different order in different devices: $a->name $b->name\n";
				return count($b->after) - count($a->after);
			} else
				return 1;
		} else if (isset($b->after[$a->name])) {
			return -1;
		}
		echo "Warning: No ordering needed between $a->name and $b->name!\n";
		return 0;
	}
	function orderCksel() {
		foreach ($this->choices AS $choice) {
			$choice->order = 0;
		}
		foreach ($this->devOrder AS $dev => $order) {
			$i = 0;
			$after = [];
			foreach ($order AS $name => $dummy) {
				$this->choices[$name]->after = $after;
				$after[$name] = true;
			}
		}
		$aftercount = 0;
		do {
			$old_aftercount = $aftercount;
			$aftercount = 0;
			foreach ($this->choices AS $name => $choice) {
				foreach ($choice->after AS $a => $t) {
					$choice->after += $this->choices[$a]->after;
					$aftercount++;
				}
			}
		} while ($aftercount != $old_aftercount);
		uasort($this->choices, ["KconfigChoice", "cmp_order"]);
	}
	function cmp_name($a, $b) { return strcmp($a->name, $b->name); }
	function convert_ms($ms) {
		if (substr($ms, -2) == "MS") {
			$ms = substr($ms, 0, -2);
		} else if (($pos = strpos($ms, "MS")) !== false) {
			$ms = str_replace("MS", ".", $ms) * 1.0;
		} else {
			echo "ERROR: invalid ms string $ms\n";
		}
		return $ms;
	}
	function convert_ck($ck, $dbg = "") {
		if (substr($ck, -3) == "KCK") {
			$ck = substr($ck, 0, -3) * 1000;
		} else if (substr($ck, -2) == "CK") {
			$ck = substr($ck, 0, -2);
		} else {
			echo "ERROR: invalid ck string $ck ($dbg)\n";
		}
		return $ck;
	}
	function cmp_sut($ao, $bo) {
		$a = explode("_", $ao->name);
		$b = explode("_", $bo->name);

		if ($a[1] != $b[1]) {
			if (count($a) == 2 || count($b) == 2) {
				if (count($a) != count($b))
					return count($a) - count($b);
				return $this->convert_ms($a[1]) - $this->convert_ms($b[1]);
			}
			return $this->convert_ck($a[1], $ao->name) - $this->convert_ck($b[1], $bo->name);
		}
		if (count($a) != count($b))
			return count($a) - count($b);
		if (count($a) == 4) {
			$r = $this->convert_ck($a[2], $ao->name) - $this->convert_ck($b[2], $bo->name);
			if ($r == 0)
				$r = $this->convert_ms($a[3]) - $this->convert_ms($b[3]);
			return $r;
		} else if (count($a) == 3) {
			return $this->convert_ms($a[2]) - $this->convert_ms($b[2]);
		} else {
			echo "ERROR: SUT invalid parts count\n";
		}
	}
	function orderByName() {
		if ($this->name == "SUT") {
			uasort($this->choices, ["KconfigChoice", "cmp_sut"]);
			return;
		} else if ($this->name == "BOOTSZ") {
			uasort($this->choices, ["KconfigChoice", "cmp_bootsz"]);
			return;
		}
		uasort($this->choices, ["KconfigChoice", "cmp_name"]);
	}
	function cmp_bootsz($a, $b) {
		$a = explode("_", $a->name);
		$b = explode("_", $b->name);
		if (substr($a[1], -1) != "W" || substr($b[1], -1) != "W") {
			echo "Illegal bootsz name ${a[1]} ${b[1]}\n";
		}
		return (substr($a[1], 0, -1) * 1) - (substr($b[1], 0, -1) * 1);
	}
	function emit() {
		$r = "choice\n";
		$r .= "\tprompt \"$this->text\"\n";
		
		if ($this->name == "CKSEL")
			$this->orderCksel();
		else
			$this->orderByName();
		$r .= $this->emitDependsAndDefaults();
		$r .= $this->addHelp();

		$r .= "\n";
		foreach ($this->choices AS $name => $choice) {
			$r .= $choice->emit();
		}
		$r .= "endchoice\n";
		$r .= "\n";
		return $r;
	}
}

class FuseKconfig {
	private $configs;

	function findChoice($name) {
		if (isset($this->configs[$name]))
			return $this->configs[$name];
		foreach ($this->configs AS $cfg) {
			$ret = $cfg->findChoice($name);
			if ($ret !== null)
				return $ret;
		}
		return null;
	}

	function addBool($name, $text, $device, $default = null, $deps = "") {
		$text = text_convert($text);

		if (!isset($this->configs[$name])) {
			$this->configs[$name] = new KconfigBool($this, $name, std_text($text, $name));
		}
		$this->configs[$name]->addDevice($device, $default, $deps, $text);
		
	}
	function addChoice($name, $text, $device, $default = null, $deps = "") {
		$text = text_convert($text);

		if (!isset($this->configs[$name])) {
			$this->configs[$name] = new KconfigChoice($this, $name, std_text($text, $name));
		}
		$this->configs[$name]->addDevice($device, $default, $deps, $text);
		return $this->configs[$name];
	}

	private $devices = [];
	function addDevice($device) { $this->devices[$device] = true; }
	function allDevices() { return $this->devices; }
	function dev_depends_list($devs) {
		if (count($devs) == count($this->devices))
			return "";

		$neg_depends = "";
		foreach ($this->devices AS $dev => $dummy) {
			if (!isset($devs[$dev])) {
				if ($neg_depends != "")
					$neg_depends .= " && ";
				$neg_depends .= "!".$dev;
			}
		}
		$normal_depends = implode(" || ", array_keys($devs));
		if (strlen($neg_depends) < strlen($normal_depends)) {
			return $neg_depends;
		}
		return $normal_depends;
	}

	function emit() {
		$r = "\n";
		foreach ($this->configs AS $config) {
			$r .= $config->emit();
		}
		return $r;
	}

	private $cpufreqs;
	function selectCpufreq($name, $hz) {
		$this->cpufreqs[$name] = $hz;
	}
	function emit_cpufreq() {
		// FIXME: integrate in new API
		$freqsel = "config CPUFREQ\n";
		$freqsel .=  "\tint\n";
		$freqsel .= "\tprompt \"Controller clock frequency\" if !CPUFREQ_IS_FIXED\n";
		$freqsel .= "\trange 100000 20000000\n";
		$freqsel .= "\tdefault 8000000\n";
		if (count($this->cpufreqs)) {
			foreach ($this->cpufreqs AS $name => $hz)
				$freqsel .= "\tdefault $hz if $name\n";
		}
		return $freqsel;
	}

	
}