<?php
// SPDX-License-Identifier: GPL-2.0
// Copyright (c) 2017-2019 Rupert Eibauer. All rights reserved.

function hweight8($x) {
	$x = ($x & 0x55) + (($x >> 1) & 0x55);
	$x = ($x & 0x33) + (($x >> 2) & 0x33);
	$x = ($x & 0x0f) + (($x >> 4) & 0x0f);
	return $x;
}

function mask_to_shift($msk) {
	$shift = 0;
	if ($msk == 0) {
		echo "Error: Mask=0\n";
		exit(1);
	}
	if (($msk & 0x0f) == 0) { $shift += 4; $msk >>= 4; }
	if (($msk & 0x03) == 0) { $shift += 2; $msk >>= 2; }
	if (($msk & 0x01) == 0) { $shift += 1; $msk >>= 1; }
	return $shift;
}


class FuseCfg {
	public $text, $name, $offset;
	public $default, $mask;
	public $shift = 0;
	function __construct($text, $name, $offset, $mask, $default) {
// 		echo "FuseCfg::construct name=$name offset=$offset mask=$mask default=$default\n";
		$this->text = $text; $this->name = $name; $this->offset = $offset;
		$this->default = $default & $mask;
		$this->mask = $mask;
		for ($i = 0; $i < 8; $i++) {
			if ($mask & (1 << $i))
				break;
		}
		$this->shift = $i;
	}
}
class BoolFuseCfg extends FuseCfg {
	function emitKconfig() {
		global $DEVICE;
		$ret = "";
		$ret .= "config ".$DEVICE."_".$this->name."\n";
		$ret .= "\tbool \"$this->text\"\n";
		$ret .= "\tdepends on $DEVICE\n";
		if ($this->default)
			$ret .= "\tdefault y\n";
		else
			$ret .= "\tdefault n\n";
		$ret .= "\n";
		return $ret;
	}
	function emitHeaderFile(&$fuse_byte, &$used_fuse_bits) {
		global $DEVICE;
		$ret = "";
		$ret .= "#ifdef CONFIG_${DEVICE}_".$this->name."\n";
		$ret .= "#define FUSE_".$this->name."_VALUE 0\n";
		$ret .= "#else\n";
		$ret .= "#define FUSE_".$this->name."_VALUE $this->mask\n";
		$ret .= "#endif\n";
		$ret .= "\n";
		$used_fuse_bits |= $this->mask;
		$fuse_byte["FUSE_".$this->name."_VALUE"] = true;
		return $ret;
	}
}
class EnumFuseCfg extends FuseCfg {
	public $options = [];
	private $optionNames = [];
	function addValue($text, $name, $value) {
// 		echo "enum $name value=$value shift=$this->shift\n";
		$i = 1;
		$suffix = "";
		while (isset($this->optionNames[$name.$suffix])) {
			$i++;
			$suffix = "_".$i;
		}
		if ($suffix != "") {
			echo "Warning: Eliminating duplicate $name with suffix $suffix\n";
		}
		$name .= $suffix;
		$this->optionNames[$name] = true;
		$this->options[$name] = [$text, $name, $value << $this->shift];
// 		$this->options[] = [$text, $name, ($value << $this->shift) ^ $this->mask];
	}
	function emitHeaderFile(&$fuse_byte, &$used_fuse_bits) {
		global $DEVICE;
// 		$ret = "// enum $this->name\n";
		$ret = "";
// 		$ret = "#define FUSE_".$this->name."
		$def = "";
		foreach ($this->options AS $opt) {
			if ($ret == "")
				$ret .= "#ifdef CONFIG_${DEVICE}_".$this->name.'_'.$opt[1]."\n";
			else
				$ret .= "#elif defined(CONFIG_${DEVICE}_".$this->name.'_'.$opt[1].")\n";

			$ret .= "#define FUSE_".$this->name."_VALUE ".$opt[2]."\n";
// 			echo "ENUM ".$this->name." _ ".$opt[1]." = ".$opt[2]."\n";
			$used_fuse_bits |= $this->mask;
		}
		$ret .= "#else\n";
		$ret .= "#error\n";
		$ret .= "#endif\n";
		$ret .= "\n";
		$fuse_byte["FUSE_".$this->name."_VALUE"] = true;
// 		$ret .= "#define FUSE_"
		return $ret;
	}
	function emitKconfig() {
		global $DEVICE;
		$ret = "";
		$default = "";
		$ret .= "\n";
		foreach ($this->options AS $opt) {
			$ret .= "config ".$DEVICE."_".$this->name."_".$opt[1]."\n";
			$ret .= "\tbool \"".$opt[0]."\"\n";
			$ret .= "\tdepends on $DEVICE\n";
			if ($this->default == $opt[2])
				$default = "\tdefault ".$DEVICE."_".$this->name.'_'.$opt[1]."\n";
			$ret .= "\n";
		}
// 		$ret = "#$this->name\nchoice\n".
		$ret = "choice\n".
		       "\tprompt \"$this->text\"\n".
		       "\tdepends on $DEVICE\n".
		       $default.
		       $ret.
		      "endchoice\n";
		$ret .= "\n";
		return $ret;
	}
}
class SplittableEnumOption {
	public $text, $name, $value, $depends = [];
	public $names = [];
	public $isDefault = false;
	static function compare($a, $b) {
		return $a->value - $b->value;
// 		return strcmp($a->name, $b->name);
	}
	function __construct($text, $name, $value, $depends = null) {
		$this->text = $text;
		$this->name = $name;
		$this->value = $value;
		if ($depends !== null)
			$this->depends[] = $depends;
	}
	function splitNames() {
		$name = explode('_', $this->name);
		$name_type = $name[0];
		$name_startup = "";
		switch($name_type) {
		case 'INTRCOSC':
			for ($i = 1; $i < count($name); $i++) {
				if (substr($name[$i], -2) == "HZ")
					$name_type .= "_".$name[$i];
				else if ($name[$i] == "DEFAULT")
					continue;
				else if (substr($name[$i], -2) == "CK" || substr($name[$i], -2) == "MS" || substr($name[$i], -3) == "MS1") {
					if ($name_startup == "")
						$name_startup = $name[$i];
					else
						$name_startup .= "_".$name[$i];
				} else {
					echo "ERROR $name_type! ".$this->name."\n";
					exit(1);
				}
				
			}
			break;
		case 'EXTRCOSC':
			if (count($name) != 5) {
				echo "ERROR $name_type! ".$this->name."\n";
				exit(1);
			}
			$name_type .= '_'.$name[1].'_'.$name[2];
			$name_startup = $name[3].'_'.$name[4];
			break;
		case 'EXTLOFXTALRES':
		case 'EXTMEDFXTALRES':
		case 'EXTHIFXTALRES':
		case 'EXTLOFXTAL':
		case 'EXTCLK':
			if (count($name) != 3 && count($name) != 4) {
				echo "ERROR $name_type! ".$this->name."\n";
				exit(1);
			}
			$name_startup = $name[1].'_'.$name[2];
			if (count($name) == 4)
				$name_startup .= "_".$name[3];
			break;
		case 'EXTFSXTAL':
			if (count($name) != 4) {
				echo "ERROR $name_type! ".$this->name."\n";
				exit(1);
			}
			$name_startup = $name[1].'_'.$name[2].'_'.$name[3];
			break;
		case 'EXTXOSC':
			if (count($name) != 6) {
				echo "ERROR $name_type! ".$this->name."\n";
				exit(1);
			}
			$name_type .= '_'.$name[1].'_'.$name[2];
			$name_startup = $name[3].'_'.$name[4].'_'.$name[5];
			break;
		default:
			echo "ERROR $name_type! ".$this->name."\n";
			echo $this->text."\n";
			print_r($name);
			exit(1);
		}
		$this->names[0] = $name_type;
		$this->names[1] = $name_startup;
	}
	function setDefault() {
		$this->isDefault = true;
	}
	function addDepend($depend) {
		$this->depends[] = $depend;
	}
}
class EnumFuseClkCfg extends FuseCfg {
	public $options = [];
	function addValue($text, $name, $value, $depends = null) {
		if (!isset($this->options[$name])) {
// 			$this->options[$name] = new SplittableEnumOption($text, $name, ($value << $this->shift) ^ $this->mask);
			$this->options[$name] = new SplittableEnumOption($text, $name, $value << $this->shift);
		}// else check?
		if ($depends !== null) {
			$this->options[$name]->addDepend($depends);
		}
	}
	function setDefault($name) {
		$this->options[$name]->setDefault();
	}
	public $subchoices;
	public $dependencies;
	function createSubChoices() {
		$this->subchoices['CLOCK'] = new EnumFuseClkCfg('Clock Type', 'CLOCK', $this->offset, 0, 0);
		$this->subchoices['STARTUP'] = new EnumFuseClkCfg('Clock Start-up time', 'STARTUP', $this->offset, 0, 0);
		foreach ($this->options AS $opt) {
			$parts = explode('; ', $opt->text);
			$default = false;
			if ($parts[count($parts)-1] == 'default value') {
				$default = true;
// 				$opt->setDefault();
				unset($parts[count($parts)-1]);
			}
			$opt->splitNames();
			$name_type = $opt->names[0];
			$name_startup = $opt->names[1];
// 			echo "full=".$opt->name."\ttype=$name_type\tstartup=$name_startup\n";
// 			echo "parts=";
// 			var_dump($parts);
// 			echo "\n";
			$desc_type = $parts[0];
// 			if (!isset($parts[1])) {
// 				echo "DATA=".$opt[0]." ".$opt[1]."\n";
// 				continue;
// 			}
			$desc_startup = explode(': ', $parts[1])[1];
// 			echo "type=$desc_type startup=$desc_startup\n";
			$this->subchoices['CLOCK']->addValue($desc_type, $name_type, 0);
			$this->subchoices['STARTUP']->addValue($desc_startup, $name_startup, 0, 'CLOCK_'.$name_type);
			if ($default) {
				$this->subchoices['CLOCK']->setDefault($name_type);
				$this->subchoices['STARTUP']->setDefault($name_startup);
			}
		}
		/* Finally, sort ourselves. */
		usort($this->options, array('SplittableEnumOption', 'compare'));
	}
	function emitHeaderFile(&$fuse_byte, &$used_fuse_bits) {
		global $DEVICE;
		$ret = "";
// 		$ret = "#define FUSE_".$this->name."
		$def = "";
		foreach ($this->options AS $opt) {
			if ($ret == "")
				$ret .= "#ifdef CONFIG_".$DEVICE."_".$this->name.'_'.$opt->name."\n";
			else
				$ret .= "#elif defined(CONFIG_".$DEVICE."_".$this->name.'_'.$opt->name.")\n";

			$ret .= "#define FUSE_".$this->name."_VALUE $opt->value\n";
		}
		$fuse_byte["FUSE_".$this->name."_VALUE"] = true;
		$ret .= "#else\n";
		$ret .= "#error\n";
		$ret .= "#endif\n";
		$ret .= "\n";
		$used_fuse_bits |= $this->mask;
// 		$ret .= "#define FUSE_"
		return $ret;
	}
	function emitKconfig() {
		global $DEVICE;
// 		echo "EnumFuseClkCfg::emitKconfig() mask=$this->mask\n";
		if ($this->mask != 0) {
			$ret = $this->subchoices['CLOCK']->emitKconfig();
			$ret .= $this->subchoices['STARTUP']->emitKconfig();
			$ret2 = "";
			foreach ($this->options AS $opt) {
				$ret .= "config ".$DEVICE."_".$this->name."_".$opt->name."\n";
// 				$ret .= "\tbool \"".$opt->text."\"\n";
				$ret .= "\tdef_bool y\n";
				$ret .= "\tdepends on ".$DEVICE."_CLOCK_".$opt->names[0]." && ".$DEVICE."_STARTUP_".$opt->names[1]."\n";
				
// 				$ret .= "\tbool \"".$opt[0]."\"\n";
// 				if (count($opt[3]))
// 					$ret .= "\tdepends on ".implode(" || ", $opt[3])."\n";
	// 			if ($this->default == $opt[2])
	// 				$default = "\tdefault ".$this->name.'_'.$opt[1]."\n";
				$ret .= "\n";
			}
// 			$ret .= $ret2;
			return $ret;
		}
		$ret = "";
		$default = "";
		$ret .= "\n";
		foreach ($this->options AS $opt) {
			$ret .= "config ".$DEVICE."_".$this->name."_".$opt->name."\n";
			$ret .= "\tbool \"".$opt->text."\"\n";
			if (count($opt->depends)) {
				$ret .= "\tdepends on ";
				$first = 1;
				foreach ($opt->depends AS $dep) {
					if (!$first) {
						$ret .= " || ";
					}
					$first = 0;
					$ret .= $DEVICE."_".$dep;
				}
				$ret .= "\n";
			}
			$optnames = explode("_", $opt->name);
			if ($this->name == "CLOCK" && $optnames[0] == "INTRCOSC") {
				global $fusecfg_cpufreq;
				$hz = 0;
				if (substr($optnames[1], -3) == "MHZ") {
					$hz = (int)($optnames[1]) * 1000000;
				} else if (substr($optnames[1], -3) == "KHZ") {
					$hz = (int)($optnames[1]) * 1000;
				} 
				if ($hz != 0) {
					$ret .= "\tselect CPUFREQ_IS_FIXED\n";
					$fusecfg_cpufreq .= "\tdefault $hz if ".$DEVICE."_$opt->name\n";
				}
					
// 				echo "$opt->name: ".(int)($optnames[1])."\n";
				/* FIXME: Set the fixed Value of CPUFREQ! */
			}
			if ($opt->isDefault)
				$default = "\tdefault ".$DEVICE."_".$this->name.'_'.$opt->name."\n";
			$ret .= "\n";
		}
// 		$ret = "#$this->name\nchoice\n".
		$ret = "choice\n".
		       "\tprompt \"$this->text\"\n".
		       "\tdepends on $DEVICE\n".
		       $default.
		       $ret.
		      "endchoice\n";
		$ret .= "\n";
		return $ret;
	}
}

function decode_number($num) {
	if (substr($num, 0, 2) == "0x")
		return hexdec(substr($num, 2));
	return $num + 0;
}

function parseRegisterGroup($node) {
	$configs = [];
	foreach ($node->childNodes AS $child) {
		if ($child->nodeName == "#text")
			continue;
		$reg_offset = decode_number($child->getAttribute('offset'));
		$reg_name = $child->getAttribute('name');
		$reg_size = decode_number($child->getAttribute('size')); // Should be 1
		$reg_initval = decode_number($child->getAttribute('initval')) ^ 0xff;
// 		echo "CHILD2: <".$child->nodeName.$attrs.">\n";
		foreach ($child->childNodes AS $bitfield) {
			if ($bitfield->nodeName == "#text")
				continue;
			$field_caption = $bitfield->getAttribute('caption');
			$field_name = $bitfield->getAttribute('name');
			$field_mask = decode_number($bitfield->getAttribute('mask'));
// 			echo "BITFIELD: name=$field_name offset=$reg_offset mask=$field_mask init=$reg_initval\n";
			if ($bitfield->hasAttribute('values')) {
				$field_values = $bitfield->getAttribute('values');
// 				echo $field_name."\n";
				if ($field_name == 'SUT_CKSEL') {
// 					echo "EnumFuseClkCfg\n";
					$configs[$field_values] = new EnumFuseClkCfg($field_caption, $field_name, $reg_offset, $field_mask, $reg_initval);
				} else {
					$configs[$field_values] = new EnumFuseCfg($field_caption, $field_name, $reg_offset, $field_mask, $reg_initval);
				}
			} else
				$configs[".".$field_name] = new BoolFuseCfg($field_caption, $field_name, $reg_offset, $field_mask, $reg_initval);
		}
	}
	return $configs;
}



/* Main program */
$Device = $argv[1];
$DEVICE = strtoupper($Device);
$device = strtolower($Device);

$file = file_get_contents("atdf/$Device.atdf");
$file = preg_replace('/<avr-tools-device-file [^>]*>/', '<avr-tools-device-file xmlns="urn:avr-tools-device-file">', $file);
$doc = new DOMDocument();
$doc->loadXML($file);

/*********************** Pre-parse XML ***********************/

$modules = $doc->getElementsByTagName("modules")->item(0)->childNodes;
$otherModules = array();
$valueGroups = array();
foreach ($modules AS $module) {
	if ($module->nodeName == "#text")
		continue;
	
	if ($module->getAttribute("name") == "FUSE") {
		$fusesModule   = $module;
	} else {
		$otherModules[] = $module;
	}
	$vgs = $module->getElementsByTagName("value-group");
	foreach ($vgs AS $vg) {
		$valueGroups[$vg->getAttribute("name")] = $vg;
	}
}

/*********************** FUSES Kconfig ***********************/

foreach ($fusesModule->childNodes AS $child) {
	if ($child->nodeName == "#text")
		continue;
	if ($child->nodeName == "register-group") {
		$groups = parseRegisterGroup($child);
	} else if ($child->nodeName == "value-group") {
		$group = $groups[$child->getAttribute('name')];
		foreach ($child->childNodes AS $valueNode) {
			if ($valueNode->nodeName == "#text")
				continue;
			$group->addValue($valueNode->getAttribute('caption'), $valueNode->getAttribute('name'), decode_number($valueNode->getAttribute('value')));
		}
	}
}

$fusecfg_cpufreq = "";
$fuseKconfig = "";
$fuseKconfig .= "comment \"$Device fuse configuration\"\n";
$fuseKconfig .= "\tdepends on $DEVICE\n";
$fuseKconfig .= "\n";
$groups = array_reverse($groups);
foreach ($groups AS $group) {
	if (get_class($group) == 'EnumFuseClkCfg') {
// 		echo "Calling createSubChoices!\n";
		$group->createSubChoices();
	}
	$fuseKconfig .= $group->emitKconfig();
}

/*********************** irqvec.h ***********************/

$interrupsNode = $doc->getElementsByTagName("interrupts")->item(0);
$irqvec_h  = "";
$irqvec_h .= "#ifndef _AVR_IRQVEC_".$DEVICE."_H_\n";
$irqvec_h .= "#define _AVR_IRQVEC_".$DEVICE."_H_\n";
$irqvec_h .= "\n";
$irqvec_h .= "/* DO NOT EDIT. generated by decode_pack.php */\n";
$irqvec_h .= "\n";
$irqvec_h1 = "";
$irqvec_h2 = "";
$max_index = 0;
foreach ($interrupsNode->childNodes AS $child) {
	if ($child->nodeName != "interrupt")
		continue;
	$index = $child->getAttribute("index");
	$name = $child->getAttribute("name");
	$irqvec_h1 .= sprintf("#define IRQ_%-12s  %3d\n", $name, $index);
	$irqvec_h2 .= sprintf("#define IRQ_NAME_%-3d %s\n", $index, $name);
	if ($max_index < $index)
		$max_index = $index;
}
$irqvec_h .= $irqvec_h1;
$irqvec_h .= "\n";
$irqvec_h .= $irqvec_h2;
$irqvec_h .= "\n";
$irqvec_h .= "#define NR_IRQS $max_index\n"; // Reset vector does not count here.
$irqvec_h .= "\n";
$irqvec_h .= "#endif /* _AVR_IRQVEC_".$DEVICE."_H_ */\n";

$vectors_h = "";
for ($i = 1; $i <= $max_index; $i++) {
	$vectors_h .= "VECTOR($i)\n";
}


/*********************** ioregs.h ***********************/

$prefix = "";
$ioregs_h  = "";
$ioregs_h .= "#ifndef _AVR_IOREGS_".$DEVICE."_H_\n";
$ioregs_h .= "#define _AVR_IOREGS_".$DEVICE."_H_\n";
$ioregs_h .= "\n";
$ioregs_h .= "/* DO NOT EDIT. generated by decode_pack.php */\n";
$ioregs_h .= "\n";
$strfmt = "%-30s";
foreach ($otherModules AS $mod) {
	$reg_groups = $mod->getElementsByTagName("register-group");
	foreach ($reg_groups AS $reg_group) {
		$ioregs_h .= "\n/* Register group ".$reg_group->getAttribute("caption")." */\n\n";
		$regs = $reg_group->getElementsByTagName("register");
		foreach ($regs AS $reg) {
			$regname = $reg->getAttribute("name");
			$offset = decode_number($reg->getAttribute("offset"));
			$regsize = $reg->getAttribute("size");
			$caption = $reg->getAttribute("caption");
			if ($caption != "") $caption = "  /* $caption */";

			$ioregs_h .= sprintf("#define $prefix$strfmt 0x%02x%s\n", $regname, $offset, $caption);
			if ($regsize > 1) {
				$suffix = array("L", "H", "HL", "HH");
				for ($i = 0; $i < $regsize; $i++) {
					$ioregs_h .= sprintf("#define $prefix$strfmt 0x%02x\n", $regname.$suffix[$i], $offset + $i);
				}
			}

			$bitfields = $reg->getElementsByTagName("bitfield");
			foreach ($bitfields AS $bitfield) {
				$name = $bitfield->getAttribute("name");
				$mask = decode_number($bitfield->getAttribute("mask"));
				if ($regname == "TCCR0") {
					if ($name == "CS02" || $name == "CS01") {
// 						echo "CS01 and CS00 detected\n";
						continue;
					}
					if ($name == "CS00") {
						$mask |= ($mask << 1) | ($mask << 2); // FIXME: Check CS02 and CS01
						$name = "CS0";
					}
				}
				$macros_used = array();
				$caption = $bitfield->getAttribute("caption");
				if (hweight8($mask) != 1) {
					if ($bitfield->hasAttribute("values")) {
						$ioregs_h .= "\n";
						$ioregs_h .= sprintf("#define $prefix$strfmt 0x%02x  /* %s */\n", $regname."_".$name."_MASK", $mask, $caption);
						$shift = mask_to_shift($mask);
						$values = $bitfield->getAttribute("values");

						$values = $valueGroups[$values]->getElementsByTagName("value");

						foreach ($values AS $value) {
							$valname = $value->getAttribute("name");
							$valval = decode_number($value->getAttribute("value"));
							$valval <<= $shift;
							$caption = $value->getAttribute("caption");
							if ($name == "CS1" || $name == "CS0" || $name == "CS2") {
								switch ($caption) {
									case "No Clock Source (Stopped)": $valname = "STOPPED"; break;
									case "Running, No Prescaling": $valname = "PS_1"; break;
									case "Running, ExtClk Tx Falling Edge": $valname = "EXTCLK_FALLING"; break;
									case "Running, ExtClk Tx Rising Edge": $valname = "EXTCLK_RISING"; break;
									default:
										if (substr($caption, 0, 13) == "Running, CLK/") {
											$valname = "PS_".substr($caption, 13);
											break;
										}
										echo "Unknown CXx value $caption\n"; exit(1);
								}
							} else if ($name == "ISC1" || $name == "ISC0") {
								switch ($caption) {
									case "Low Level of INTX": $valname = "LOW"; break;
									case "Any Logical Change of INTX": $valname = "ANY"; break;
									case "Any Logical Change in INTX": $valname = "ANY"; break;
									case "Falling Edge of INTX": $valname = "FALLING"; break;
									case "Rising Edge of INTX": $valname = "RISING"; break;
									default: echo "Unknown ISCx value \"$caption\"\n"; exit(1);
								}
							} else if ($name == "REFS") {
								switch ($caption) {
									case "AREF, Internal Vref turned off": $valname = "AREF"; break;
									case "AVCC with external capacitor at AREF pin": $valname = "AVCC"; break;
									case "Reserved": break;
									case "Internal 2.56V Voltage Reference with external capacitor at AREF pin": $valname = "2V56"; break;
									case "Internal 1.1V Voltage Reference with external capacitor at AREF pin": $valname = "1V1"; break;
									default: echo "Unknown REFS value $caption\n"; exit(1);
								}
							} else if ($name == "ADPS" || $name == "TWPS") {
								if ((string)(int)$caption !== (string)$caption) {
									echo "Unknown ADPS/TWPS value $caption\n";
									exit(1);
								}
								$valname = $caption;
							} else if ($name == "WDP") {
								if (substr($caption, 0, 18) !== "Oscillator Cycles ") {
									echo "Unknown ADPS/TWPS value $caption\n";
									exit(1);
								}
								$valname = substr($caption, 18);
							} else if ($name == "UPM") {
								switch ($caption) {
									case "Disabled": $valname = "DISABLED"; break;
									case "Enabled, Even Parity": $valname = "EVEN"; break;
									case "Enabled, Odd Parity": $valname = "ODD"; break;
									case "Reserved": break;
									default: echo "Unknown UPM value $caption\n"; exit(1);
								}
							} else if ($name == "ACIS") {
								switch ($caption) {
									case "Interrupt on Toggle": $valname = "TOGGLE"; break;
									case "Interrupt on Falling Edge": $valname = "FALLING"; break;
									case "Interrupt on Rising Edge": $valname = "RISING"; break;
									case "Reserved": break;
									default: echo "Unknown UPM value $caption\n"; exit(1);
								}
							} else if ($name == "SPR") {
								if (substr($caption, 0, 5) !== "fosc/") {
									echo "Unknown SPR value $caption\n";
									exit(1);
								}
								$valname = substr($caption, 5);
							}
							$valname = $regname."_".$name."_".$valname;
							if (isset($macros_used[$valname])) {
								$dup = 0;
								while (isset($macros_used[$valname."_DUP$dup"])) {
								}
								$valname .= "_DUP$dup";
							}
							$macros_used[$valname] = true;
							$ioregs_h .= sprintf("#define $prefix$strfmt 0x%02x  /* %s */\n", $valname, $valval, $caption);
						}
					} else {
						$ioregs_h .= sprintf("#define $prefix$strfmt 0x%02x  /* %s */\n", $regname."_".$name."_MASK", $mask, $caption);
					}
				} else {
					$ioregs_h .= sprintf("#define $prefix$strfmt 0x%02x  /* %s */\n", $regname.'_'.$name, $mask, $caption);
					$ioregs_h .= sprintf("#define $prefix$strfmt 0x%02x\n", $regname.'_'.$name."_BIT", mask_to_shift($mask));
				}
			}
			$ioregs_h .= "\n";
		}
	}
}
$ioregs_h .= "#endif /* _AVR_IOREGS_".$DEVICE."_H_ */\n";

/*********************** limits.h ***********************/
$num_fuse_bytes = 0;

$memorySegmentNodes = $doc->getElementsByTagName("memory-segment");
$limits_h  = "";
$limits_h .= "#ifndef _AVR_LIMITS_".$DEVICE."_H_\n";
$limits_h .= "#define _AVR_LIMITS_".$DEVICE."_H_\n";
$limits_h .= "\n";
$limits_h .= "/* DO NOT EDIT. generated by decode_pack.php */\n";
$limits_h .= "\n";
foreach ($memorySegmentNodes AS $child) {
	if ($child->nodeName != "memory-segment")
		continue;
	$name = $child->getAttribute("name");
	if ($name != "IRAM" && $name != "FLASH" && $name != "EEPROM" && $name != "FUSES" && $name != "LOCKBITS")
		continue;
	if ($name == "IRAM")
		$name = "RAM";

	$start_name = $name."_START";
	$end_name = $name."_END";
	$start_value = decode_number($child->getAttribute("start"));
	$end_value = $start_value + decode_number($child->getAttribute("size")) - 1;
	if ($name == "FUSES")
		$num_fuse_bytes = $end_value + 1;

	$limits_h .= sprintf("#define %-15s 0x%x\n", $start_name, $start_value);
	$limits_h .= sprintf("#define %-15s 0x%x\n", $end_name, $end_value);
	$limits_h .= "\n";
	if ($name == "FLASH" || $name == "EEPROM") {
		$pagesize = decode_number($child->getAttribute("pagesize"));
		$limits_h .= sprintf("#define %-15s 0x%x\n", $name."_PAGESIZE", $pagesize);
		$limits_h .= "\n";
	}
}
$limits_h .= "#endif /* _AVR_LIMITS_".$DEVICE."_H_ */\n";

/*********************** fusecalc.h ***********************/

$fusecalc_h  = "#ifndef _AVR_FUSECALC_".$DEVICE."_H\n";
$fusecalc_h .= "#define _AVR_FUSECALC_".$DEVICE."_H_\n";
$fusecalc_h .= "\n";
$fusecalc_h .= "/* DO NOT EDIT. generated by decode_pack.php */\n";
$fusecalc_h .= "\n";
$fuse_bytes = [];
$used_fuse_bits = [];
for ($i = 0; $i < $num_fuse_bytes; $i++) {
	$fuse_bytes[] = [];
	$used_fuse_bits[] = 0;
}
foreach ($groups AS $group) {
	if ($group->offset >= $num_fuse_bytes) {
		echo "ILLEGAL FUSE OFFSET $group->offset!\n";
		exit(1);
	}
	$fusecalc_h .= $group->emitHeaderFile($fuse_bytes[$group->offset], $used_fuse_bits[$group->offset]);
}
for ($i = 0; $i < $num_fuse_bytes; $i++) {
	$mask = 0xff & ~$used_fuse_bits[$i];
	$value = sprintf("0x%02x", $mask);
	if (count($fuse_bytes[$i]))
		$value = "(".$value." | ".implode(' | ', array_keys($fuse_bytes[$i])).")";
	
	$fusecalc_h .= "#define FUSE_BYTE_$i $value\n";
}
$fusecalc_h .= "\n";
$fusecalc_h .= "#endif /* _AVR_FUSECALC_".$DEVICE."_H_ */\n";

/*********************** Finally, write everything to the files ... ***********************/

file_put_contents("devs/allavr.Kconfig", "config $DEVICE\n\tbool \"$Device\"\n\n", FILE_APPEND);
file_put_contents("devs/allfuses.Kconfig", "source \"arch/avr/devs/$device/Kconfig\"\n", FILE_APPEND);
file_put_contents("devs/cpufreq.Kconfig", $fusecfg_cpufreq, FILE_APPEND);
file_put_contents("devs/devices.Makefile", "cpu-$(CONFIG_$DEVICE) := $device\n", FILE_APPEND);

mkdir("devs/$device/include/asm", 0755, true);
file_put_contents("devs/$device/Kconfig", $fuseKconfig);
file_put_contents("devs/$device/include/asm/fusecalc.h", $fusecalc_h);
file_put_contents("devs/$device/include/asm/irqvec.h", $irqvec_h);
file_put_contents("devs/$device/include/asm/ioregs.h", $ioregs_h);
file_put_contents("devs/$device/include/asm/vectors.h", $vectors_h);
file_put_contents("devs/$device/include/asm/limits.h", $limits_h);

exit(0);
